import { ComponentQuality, FilterDef, contributionPoints } from "./resolveScore10";

export type FilterContribution = {
  id: string;
  key: string;
  label: string;
  weight: number;
  maxScore: number;
  score10: number;
  points: number;
  ratio: number;
};

export type FilterEvaluation = {
  key: string;
  totalPoints: number;
  maxPoints: number;
  normalizedScore: number | null;
  breakdown: FilterContribution[];
};

export type FilterEvaluationOptions = {
  base?: number;
  scale?: number;
};

const clamp = (value: number, min: number, max: number) => Math.max(min, Math.min(max, value));

function toContribution(filter: FilterDef, bike: any, quality: ComponentQuality): FilterContribution | null {
  const id = filter.id ? String(filter.id) : null;
  const key = filter.key ? String(filter.key) : id;
  if (!id || !key) {
    return null;
  }

  const weight = Number.isFinite(filter.weight) ? Number(filter.weight) : 0;
  if (weight <= 0) {
    return null;
  }

  const maxScore = Number.isFinite(filter.maxScore) ? Number(filter.maxScore) : 10;
  if (maxScore <= 0) {
    return null;
  }

  const { score10, points } = contributionPoints(filter, bike, quality);
  const normalizedPoints = Number.isFinite(points) ? points : 0;
  const safeScore10 = Number.isFinite(score10) ? score10 : 0;
  const ratio = weight > 0 ? normalizedPoints / weight : 0;

  return {
    id,
    key,
    label: filter.label ? String(filter.label) : key,
    weight,
    maxScore,
    score10: clamp(safeScore10, 0, maxScore),
    points: clamp(normalizedPoints, 0, weight),
    ratio: clamp(ratio, 0, 1),
  };
}

export function evaluateFilter(
  filterKey: string,
  filters: FilterDef[],
  bike: any,
  quality: ComponentQuality,
  options: FilterEvaluationOptions = {}
): FilterEvaluation {
  const breakdown: FilterContribution[] = [];
  let totalPoints = 0;
  let maxPoints = 0;

  filters.forEach(filter => {
    const contribution = toContribution(filter, bike, quality);
    if (!contribution) {
      return;
    }
    breakdown.push(contribution);
    totalPoints += contribution.points;
    maxPoints += contribution.weight;
  });

  const normalizedScore = computeNormalizedScore(totalPoints, maxPoints, options);

  return {
    key: filterKey,
    totalPoints,
    maxPoints,
    normalizedScore,
    breakdown,
  };
}

function computeNormalizedScore(
  totalPoints: number,
  maxPoints: number,
  options: FilterEvaluationOptions
): number | null {
  if (!Number.isFinite(totalPoints) || !Number.isFinite(maxPoints) || maxPoints <= 0) {
    return null;
  }

  const ratio = totalPoints <= 0 ? 0 : totalPoints / maxPoints;
  const base = options.base ?? 0;
  const scale = options.scale ?? 10;
  const maxValue = base + scale;
  return Number(clamp(base + ratio * scale, base, maxValue).toFixed(2));
}

export function evaluateFilters(
  definitions: Record<string, FilterDef[]>,
  bike: any,
  quality: ComponentQuality,
  options: Record<string, FilterEvaluationOptions> = {}
): Record<string, FilterEvaluation> {
  const result: Record<string, FilterEvaluation> = {};
  if (!definitions || typeof definitions !== "object") {
    return result;
  }

  Object.entries(definitions).forEach(([key, filters]) => {
    if (!Array.isArray(filters) || !filters.length) {
      return;
    }
    result[key] = evaluateFilter(key, filters, bike, quality, options[key] ?? {});
  });

  return result;
}
