"""Rebuild the component quality lookup tables.

This script consumes the bikes dataset and the filter definitions to
reconstruct the ``scoring/component_quality.json`` file. The generated
tables follow the guidance contained in the ``explanation`` section of
each filter criterion. The goal is to provide an up-to-date and
comprehensive mapping for every component referenced by the scoring
system so that no bike value is left without an associated quality
score.
"""

from __future__ import annotations

import json
import math
import pathlib
import re
import unicodedata
from dataclasses import dataclass
from typing import Any, Callable, Dict, Iterable, List, Mapping, Optional, Tuple


ROOT = pathlib.Path(__file__).resolve().parents[1]


def load_json(path: pathlib.Path) -> Any:
    with path.open("r", encoding="utf-8") as fh:
        return json.load(fh)


def normalize_label(value: str) -> str:
    """Replicate the normalisation used by the scoring helpers."""

    normalized = unicodedata.normalize("NFKD", value)
    normalized = normalized.encode("ascii", "ignore").decode("ascii")
    normalized = normalized.lower()
    normalized = re.sub(r"[^a-z0-9]+", " ", normalized)
    return normalized.strip()


def format_numeric(value: Optional[float]) -> Optional[str]:
    if value is None:
        return None
    if math.isfinite(value):
        if abs(value - round(value)) < 1e-6:
            return str(int(round(value)))
        return f"{value:.2f}".rstrip("0").rstrip(".")
    return None


def clamp(value: float, minimum: float, maximum: float) -> float:
    return max(minimum, min(maximum, value))


def extract_tyre_width(raw: Any) -> Optional[float]:
    if raw is None:
        return None
    if isinstance(raw, (list, tuple, set)):
        for item in raw:
            width = extract_tyre_width(item)
            if width is not None:
                return width
        return None
    if isinstance(raw, (int, float)):
        if math.isfinite(raw):
            return float(raw)
        return None
    text = str(raw).lower().replace(",", ".")
    match = re.search(r"(\d+(?:\.\d+)?)\s*mm", text)
    if match:
        return float(match.group(1))
    match = re.search(r"x\s*(\d+(?:\.\d+)?)", text)
    if match:
        width = float(match.group(1))
        if width > 10:
            return width
        return width * 25.4
    match = re.search(r"(\d+(?:\.\d+)?)\s*(?:po|in|inch)", text)
    if match:
        return float(match.group(1)) * 25.4
    numbers = re.findall(r"\d+(?:\.\d+)?", text)
    if numbers:
        candidate = float(numbers[-1])
        if candidate <= 10:
            return candidate * 25.4
        return candidate
    return None


def classify_tyre(label: Any, fallback_category: Optional[str] = None) -> Optional[str]:
    width = extract_tyre_width(label)
    if width is not None:
        if width >= 95:
            return "fatbike"
        if width >= 50:
            return "mountain"
        return "road"

    candidates: List[str] = []
    if isinstance(label, str):
        candidates.append(label)
    if fallback_category:
        candidates.append(fallback_category)

    fat_keywords = ("fat", "snow", "juggernaut", "minion fbf", "dillinger")
    mountain_keywords = (
        "mountain",
        "mtb",
        "trail",
        "enduro",
        "dh",
        "dhr",
        "aggressor",
        "assegai",
        "crampon",
        "knob",
    )
    road_keywords = (
        "road",
        "pavement",
        "slick",
        "urbain",
        "city",
        "street",
        "commuter",
        "gravel",
        "marathon",
    )

    for candidate in candidates:
        label_norm = normalize_label(candidate)
        if not label_norm:
            continue
        if any(keyword in label_norm for keyword in fat_keywords):
            return "fatbike"
        if any(keyword in label_norm for keyword in mountain_keywords):
            return "mountain"
        if any(keyword in label_norm for keyword in road_keywords):
            return "road"

    return None


def parse_speeds(value: Any, fallback_label: Optional[str] = None) -> Optional[int]:
    if isinstance(value, (int, float)) and value > 0:
        return int(round(value))
    for source in (value, fallback_label):
        if not isinstance(source, str):
            continue
        numbers = re.findall(r"(\d+)(?:\s*vitesses|\s*v)\b", source.lower())
        if numbers:
            return int(numbers[-1])
        digits = re.findall(r"\d+", source)
        if digits:
            candidate = int(digits[-1])
            if 1 <= candidate <= 14:
                return candidate
    return None


def score_battery_capacity(capacity_wh: Optional[float], *, cold: bool = False) -> Optional[float]:
    if capacity_wh is None:
        return None
    if capacity_wh <= 0:
        return 0.0
    # At very low temperature the effective capacity drops; give a slight penalty.
    target = 1440.0
    ratio = clamp(capacity_wh / target, 0.0, 1.0)
    score = ratio * 10.0
    if cold:
        score *= 0.9
    return round(score, 2)


def score_motor_type(value: Any, *, on_road: bool = False, private_use: bool = False) -> Optional[float]:
    if value is None:
        return None
    label = normalize_label(str(value))
    if "mid" in label and "drive" in label:
        if private_use:
            return 5.0
        return 10.0
    if "rear" in label or "arriere" in label or "hub" in label:
        if private_use:
            return 10.0
        return 0.0
    if "front" in label or "avant" in label:
        if private_use:
            return 8.0
        return 2.0
    return 5.0 if private_use else 7.0


def score_motor_brand_quality(value: Any) -> Optional[float]:
    if value is None:
        return None
    label = normalize_label(str(value))
    if not label:
        return None
    if any(keyword in label for keyword in ("bosch", "specialized", "brose")):
        return 9.5
    if "yamaha" in label or "shimano" in label or "steps" in label:
        return 8.5
    if "mah" in label or "mahle" in label or "fazua" in label:
        return 8.0
    if "bafang" in label:
        if "m6" in label or "m620" in label or "m600" in label:
            return 7.5
        return 6.5
    if "dapu" in label or "xiongda" in label:
        return 5.0
    if "unknown" in label or "inconnu" in label or "generic" in label:
        return 2.0
    return 6.0


def score_motor_power(value: Optional[float], *, favour_low: bool = False, minimum: float = 250.0) -> Optional[float]:
    if value is None:
        return None
    watts = float(value)
    if favour_low:
        if watts <= minimum:
            return 10.0
        if watts >= 1000.0:
            return 0.0
        ratio = (watts - minimum) / (1000.0 - minimum)
        return round(10.0 * (1.0 - ratio), 2)
    # Otherwise more power is better.
    if watts <= minimum:
        return 5.0
    if watts >= 1000.0:
        return 10.0
    ratio = (watts - minimum) / (1000.0 - minimum)
    return round(5.0 + ratio * 5.0, 2)


def score_motor_torque(value: Optional[float]) -> Optional[float]:
    if value is None:
        return None
    torque = float(value)
    if torque <= 40.0:
        return 1.0
    if torque >= 160.0:
        return 10.0
    ratio = (torque - 40.0) / (160.0 - 40.0)
    return round(1.0 + ratio * 9.0, 2)


def score_transmission(speeds: Optional[int]) -> Optional[float]:
    if speeds is None:
        return None
    if speeds <= 1:
        return 1.0
    if speeds >= 12:
        return 10.0
    ratio = (speeds - 1) / (12 - 1)
    return round(1.0 + ratio * 9.0, 2)


def score_range_extender(value: Any) -> Optional[float]:
    if value is None:
        return 0.0
    label = normalize_label(str(value))
    if not label or label in {"aucune", "aucun", "none", "sans"}:
        return 0.0
    return 10.0


def score_frame_suspension(value: Any) -> Optional[float]:
    if value is None:
        return None
    label = normalize_label(str(value))
    if any(token in label for token in ("sans", "rigide", "none", "no")):
        return 10.0
    if "simple" in label or "hardtail" in label or "front" in label:
        return 5.0
    if "double" in label or "full" in label or "suspendu" in label:
        return 0.0
    return 5.0


def score_tyres_autonomy(label: Any, fallback_category: Optional[str]) -> Optional[float]:
    category = classify_tyre(label, fallback_category)
    if category == "fatbike":
        return 0.0
    if category == "mountain":
        return 4.0
    if category == "road":
        return 10.0
    return 6.0


def score_tyres_road(label: Any, fallback_category: Optional[str]) -> Optional[float]:
    category = classify_tyre(label, fallback_category)
    if category == "fatbike":
        return 0.0
    if category == "mountain":
        return 5.0
    if category == "road":
        return 10.0
    return 6.0


def score_tyres_offroad(label: Any, fallback_category: Optional[str]) -> Optional[float]:
    category = classify_tyre(label, fallback_category)
    if category == "road":
        return 0.0
    if category == "fatbike":
        return 8.0
    if category == "mountain":
        return 9.0
    return 5.0


def score_tyres_winter(label: Any, fallback_category: Optional[str]) -> Optional[float]:
    category = classify_tyre(label, fallback_category)
    if category == "road":
        return 0.0
    if category == "fatbike":
        return 9.0
    if category == "mountain":
        return 4.0
    return 5.0


def score_weight(value: Optional[float], *, light_best: bool = True) -> Optional[float]:
    if value is None:
        return None
    weight = float(value)
    if not light_best:
        if weight >= 35:
            return 10.0
        if weight <= 20:
            return 3.0
        ratio = (weight - 20.0) / (35.0 - 20.0)
        return round(3.0 + ratio * 7.0, 2)

    if weight <= 18.0:
        return 10.0
    if weight >= 35.0:
        return 1.0
    ratio = (weight - 18.0) / (35.0 - 18.0)
    return round(10.0 - ratio * 9.0, 2)


def score_category_position(bike: Mapping[str, Any]) -> Optional[float]:
    label = classify_tyre(bike.get("tyres_label"), bike.get("category"))
    if label == "fatbike":
        return 0.0
    if label == "mountain":
        return 5.0
    if label == "road":
        return 10.0
    category = normalize_label(str(bike.get("category", "")))
    if "utilitaire" in category or "urbain" in category:
        return 9.0
    if "montagne" in category or "trail" in category:
        return 5.0
    return 7.0


def score_display_quality(value: Any) -> Optional[float]:
    if value is None:
        return None
    label = normalize_label(str(value))
    if not label:
        return None
    if any(keyword in label for keyword in ("inconnu", "generic")):
        return 1.0
    if "led" in label and "display" not in label and "lcd" not in label:
        return 3.0
    if any(keyword in label for keyword in ("lcd", "led display", "simple")):
        return 4.0
    if any(keyword in label for keyword in ("kiox", "nyon", "intuvia", "purion", "color")):
        return 8.0
    if any(keyword in label for keyword in ("bluetooth", "connect", "smart", "touch")):
        return 10.0
    return 6.0


def score_controller_quality(value: Any) -> Optional[float]:
    if value is None:
        return None
    label = normalize_label(str(value))
    if not label:
        return None
    if any(keyword in label for keyword in ("bosch", "shimano", "brose", "yamaha")):
        return 8.5
    if "fazua" in label or "mahle" in label or "specialized" in label:
        return 8.0
    if "inconnu" in label or "generic" in label:
        return 3.0
    if "bafang" in label:
        return 6.5
    return 5.0


def score_boolean(flag: Any, *, truthy: float = 10.0, falsy: float = 0.0) -> Optional[float]:
    if flag is None:
        return None
    if isinstance(flag, str):
        label = normalize_label(flag)
        if label in {"oui", "yes", "true"}:
            return truthy
        if label in {"non", "no", "false"}:
            return falsy
    if isinstance(flag, bool):
        return truthy if flag else falsy
    return truthy if flag else falsy


def score_derailleur_quality(value: Any) -> Optional[float]:
    if value is None:
        return None
    label = normalize_label(str(value))
    if not label:
        return None
    tiers = [
        ("tourney", 2.0),
        ("altus", 3.0),
        ("acera", 4.0),
        ("alivio", 4.5),
        ("cues", 5.5),
        ("deore", 6.5),
        ("slx", 7.0),
        ("xt", 8.0),
        ("xtr", 9.5),
        ("grx", 7.0),
        ("force", 8.5),
        ("rival", 7.5),
        ("apex", 6.0),
        ("gx", 7.5),
        ("sx", 5.0),
        ("nx", 6.0),
        ("x0", 9.0),
        ("xx", 9.5),
        ("eagle", 7.5),
        ("automatic", 6.0),
        ("enviolo", 6.0),
        ("rohloff", 9.5),
    ]
    for keyword, score in tiers:
        if keyword in label:
            return score
    if "microshift" in label:
        return 4.5
    if "sram" in label or "shimano" in label:
        return 6.0
    return 5.0


def score_brakes(value: Any, rotor: Optional[float]) -> Optional[float]:
    if value is None and rotor is None:
        return None
    score = 0.0
    label = normalize_label(str(value)) if value is not None else ""
    if "mechan" in label:
        score = 3.0
    elif "magura" in label or "code" in label or "saint" in label:
        score = 9.0
    elif "four piston" in label or "4 piston" in label:
        score = 7.0
    elif "tektro" in label:
        score = 5.0
    elif "shimano" in label:
        score = 6.5
    elif "sram" in label:
        score = 6.5
    elif "hyd" in label:
        score = 5.5
    elif "drum" in label:
        score = 2.0
    else:
        score = 4.5

    if rotor is not None:
        if rotor >= 220:
            score = max(score, 9.5)
        elif rotor >= 200:
            score = max(score, 8.0)
        elif rotor >= 180:
            score = max(score, 6.0)
        elif rotor >= 160:
            score = max(score, 4.5)
    return round(clamp(score, 0.0, 10.0), 2)


def score_suspension(value: Any, *, allow_none_high: bool = False) -> Optional[float]:
    if value is None:
        return 10.0 if allow_none_high else 0.0
    label = normalize_label(str(value))
    if not label:
        return 5.0
    if any(keyword in label for keyword in ("rigid", "none", "sans")):
        return 10.0 if allow_none_high else 0.0
    if any(keyword in label for keyword in ("suntour", "sr suntour", "x fusion")):
        return 4.0
    if any(keyword in label for keyword in ("rockshox recon", "rockshox 35", "fox rhythm", "marzocchi")):
        return 6.5
    if any(keyword in label for keyword in ("rockshox lyrik", "fox 36", "fox 38", "zeb", "sid", "factory", "ultimate")):
        return 9.0
    if "coil" in label:
        return 8.0
    return 6.0


def score_rotor_size(value: Optional[float]) -> Optional[float]:
    if value is None:
        return None
    rotor = float(value)
    if rotor <= 160:
        return 2.0
    if rotor >= 220:
        return 10.0
    ratio = (rotor - 160.0) / (220.0 - 160.0)
    return round(2.0 + ratio * 8.0, 2)


def score_battery_cells(value: Any) -> Optional[float]:
    if value is None:
        return 0.0
    label = normalize_label(str(value))
    if not label:
        return 0.0
    if any(keyword in label for keyword in ("samsung", "lg", "panasonic", "sony")):
        return 10.0
    if "inconnu" in label or "generic" in label:
        return 0.0
    return 7.0


def score_category_private(value: Any) -> Optional[float]:
    return score_motor_type(value, private_use=True)


def collect_raw_value(bike: Mapping[str, Any], fields: Iterable[str]) -> Any:
    for field in fields:
        if field in bike and bike[field] is not None:
            return bike[field]
    return None


@dataclass
class CriterionConfig:
    fields: Tuple[str, ...]
    score: Callable[[Any, Mapping[str, Any]], Optional[float]]
    label: Optional[Callable[[Any, Mapping[str, Any]], Optional[str]]] = None


def default_formatter(value: Any) -> Optional[str]:
    if value is None:
        return None
    if isinstance(value, bool):
        return "true" if value else "false"
    if isinstance(value, (int, float)):
        return format_numeric(float(value))
    return str(value)


def bool_formatter(value: Any) -> Optional[str]:
    if isinstance(value, bool):
        return "true" if value else "false"
    if value is None:
        return "false"
    label = normalize_label(str(value))
    if label in {"oui", "yes", "true"}:
        return "true"
    return "false"


CRITERIA_CONFIG: Dict[Tuple[str, str], CriterionConfig] = {
    ("autonomy-01", "cmp-002"): CriterionConfig(
        ("batterie_cap_wh",), lambda raw, _: score_battery_capacity(raw)
    ),
    ("autonomy-02", "cmp-006"): CriterionConfig(
        ("battery_cells",), lambda raw, _: score_battery_cells(raw)
    ),
    ("autonomy-03", "cmp-032"): CriterionConfig(
        ("range_extender",), lambda raw, _: score_range_extender(raw)
    ),
    ("autonomy-04", "cmp-029"): CriterionConfig(
        ("motor_type",), lambda raw, _: score_motor_type(raw)
    ),
    ("autonomy-05", "cmp-027"): CriterionConfig(
        ("motor_power_w",), lambda raw, _: score_motor_power(raw, favour_low=True)
    ),
    ("autonomy-06", "cmp-021"): CriterionConfig(
        ("frame_suspension",), lambda raw, _: score_frame_suspension(raw)
    ),
    ("autonomy-07", "cmp-039"): CriterionConfig(
        ("tyres_label", "tyres"),
        lambda raw, bike: score_tyres_autonomy(raw, bike.get("category")),
    ),
    ("autonomy-08", "cmp-016"): CriterionConfig(
        ("derailleur_speeds", "derailleur_label"),
        lambda raw, bike: score_transmission(parse_speeds(raw, bike.get("derailleur_label"))),
        label=lambda raw, _: format_numeric(parse_speeds(raw)),
    ),
    ("road_performance-01", "cmp-029"): CriterionConfig(
        ("motor_type",), lambda raw, _: score_motor_type(raw, on_road=True)
    ),
    ("road_performance-02", "cmp-027"): CriterionConfig(
        ("motor_power_w",), lambda raw, _: score_motor_power(raw)
    ),
    ("road_performance-03", "cmp-028"): CriterionConfig(
        ("motor_torque_nm",), lambda raw, _: score_motor_torque(raw)
    ),
    ("road_performance-04", "cmp-016"): CriterionConfig(
        ("derailleur_speeds", "derailleur_label"),
        lambda raw, bike: score_transmission(parse_speeds(raw, bike.get("derailleur_label"))),
        label=lambda raw, _: format_numeric(parse_speeds(raw)),
    ),
    ("road_performance-05", "cmp-039"): CriterionConfig(
        ("tyres_label", "tyres"),
        lambda raw, bike: score_tyres_road(raw, bike.get("category")),
    ),
    ("road_performance-06", "cmp-041"): CriterionConfig(
        ("weight_kg",), lambda raw, _: score_weight(raw)
    ),
    ("road_performance-07", "cmp-033"): CriterionConfig(
        ("category", "tyres_label"), lambda raw, bike: score_category_position(bike)
    ),
    ("road_performance-08", "cmp-017"): CriterionConfig(
        ("display",), lambda raw, _: score_display_quality(raw)
    ),
    ("road_performance-09", "cmp-038"): CriterionConfig(
        ("throttle",), lambda raw, _: score_boolean(raw), label=lambda raw, _: bool_formatter(raw)
    ),
    ("offroad_performance-01", "cmp-029"): CriterionConfig(
        ("motor_type",), lambda raw, _: score_motor_type(raw)
    ),
    ("offroad_performance-02", "cmp-028"): CriterionConfig(
        ("motor_torque_nm",), lambda raw, _: score_motor_torque(raw)
    ),
    ("offroad_performance-03", "cmp-015"): CriterionConfig(
        ("derailleur_label",), lambda raw, _: score_derailleur_quality(raw)
    ),
    ("offroad_performance-04", "cmp-037"): CriterionConfig(
        ("suspension_avant",), lambda raw, _: score_suspension(raw)
    ),
    ("offroad_performance-05", "cmp-036"): CriterionConfig(
        ("suspension_arriere",), lambda raw, _: score_suspension(raw)
    ),
    ("offroad_performance-06", "cmp-008"): CriterionConfig(
        ("brakes_label",), lambda raw, bike: score_brakes(raw, bike.get("brakes_rotor_mm"))
    ),
    ("offroad_performance-07", "cmp-039"): CriterionConfig(
        ("tyres_label", "tyres"),
        lambda raw, bike: score_tyres_offroad(raw, bike.get("category")),
    ),
    ("offroad_performance-08", "cmp-041"): CriterionConfig(
        ("weight_kg",), lambda raw, _: score_weight(raw)
    ),
    ("offroad_performance-09", "cmp-038"): CriterionConfig(
        ("throttle",), lambda raw, _: score_boolean(raw), label=lambda raw, _: bool_formatter(raw)
    ),
    ("offroad_performance-10", "cmp-009"): CriterionConfig(
        ("brakes_rotor_mm",), lambda raw, _: score_rotor_size(raw)
    ),
    ("private_performance-01", "cmp-028"): CriterionConfig(
        ("motor_torque_nm",), lambda raw, _: score_motor_torque(raw)
    ),
    ("private_performance-05", "cmp-039"): CriterionConfig(
        ("tyres_label", "tyres"),
        lambda raw, bike: score_tyres_offroad(raw, bike.get("category")),
    ),
    ("private_performance-06", "cmp-008"): CriterionConfig(
        ("brakes_label",), lambda raw, bike: score_brakes(raw, bike.get("brakes_rotor_mm"))
    ),
    ("private_performance-07", "cmp-037"): CriterionConfig(
        ("suspension_avant",), lambda raw, _: score_suspension(raw)
    ),
    ("private_performance-08", "cmp-036"): CriterionConfig(
        ("suspension_arriere",), lambda raw, _: score_suspension(raw)
    ),
    ("private_performance-09", "cmp-029"): CriterionConfig(
        ("motor_type",), lambda raw, _: score_category_private(raw)
    ),
    ("private_performance-10", "cmp-027"): CriterionConfig(
        ("motor_power_w",), lambda raw, _: score_motor_power(raw)
    ),
    ("utility_comfort-01", "cmp-035"): CriterionConfig(
        ("stem_adjustable",), lambda raw, _: score_boolean(raw), label=lambda raw, _: bool_formatter(raw)
    ),
    ("utility_comfort-02", "cmp-023"): CriterionConfig(
        ("lights_front", "lights_rear"),
        lambda raw, bike: score_boolean(bool(bike.get("lights_front")) and bool(bike.get("lights_rear"))),
        label=lambda raw, bike: "true"
        if bool(bike.get("lights_front")) and bool(bike.get("lights_rear"))
        else "false",
    ),
    ("utility_comfort-03", "cmp-019"): CriterionConfig(
        ("fenders_front", "fenders_rear"),
        lambda raw, bike: score_boolean(bool(bike.get("fenders_front")) and bool(bike.get("fenders_rear"))),
        label=lambda raw, bike: "true"
        if bool(bike.get("fenders_front")) and bool(bike.get("fenders_rear"))
        else "false",
    ),
    ("utility_comfort-04", "cmp-031"): CriterionConfig(
        ("rack_rear",), lambda raw, _: score_boolean(raw), label=lambda raw, _: bool_formatter(raw)
    ),
    ("utility_comfort-05", "cmp-038"): CriterionConfig(
        ("throttle",), lambda raw, _: score_boolean(raw), label=lambda raw, _: bool_formatter(raw)
    ),
    ("utility_comfort-06", "cmp-023"): CriterionConfig(
        ("lights_front",), lambda raw, _: score_boolean(raw), label=lambda raw, _: bool_formatter(raw)
    ),
    ("utility_comfort-07", "cmp-031"): CriterionConfig(
        ("rack_front",), lambda raw, _: score_boolean(raw), label=lambda raw, _: bool_formatter(raw)
    ),
    ("winter_performance-01", "cmp-028"): CriterionConfig(
        ("motor_torque_nm",), lambda raw, _: score_motor_torque(raw)
    ),
    ("winter_performance-02", "cmp-039"): CriterionConfig(
        ("tyres_label", "tyres"),
        lambda raw, bike: score_tyres_winter(raw, bike.get("category")),
    ),
    ("winter_performance-03", "cmp-008"): CriterionConfig(
        ("brakes_label",), lambda raw, bike: score_brakes(raw, bike.get("brakes_rotor_mm"))
    ),
    ("winter_performance-04", "cmp-002"): CriterionConfig(
        ("batterie_cap_wh",), lambda raw, _: score_battery_capacity(raw, cold=True)
    ),
    ("winter_performance-09", "cmp-041"): CriterionConfig(
        ("weight_kg",), lambda raw, _: score_weight(raw, light_best=False)
    ),
    ("reliability-01", "cmp-026"): CriterionConfig(
        ("motor_brand",), lambda raw, _: score_motor_brand_quality(raw)
    ),
    ("reliability-02", "cmp-014"): CriterionConfig(
        ("controller",), lambda raw, _: score_controller_quality(raw)
    ),
    ("reliability-03", "cmp-017"): CriterionConfig(
        ("display",), lambda raw, _: score_display_quality(raw)
    ),
    ("reliability-04", "cmp-015"): CriterionConfig(
        ("derailleur_label",), lambda raw, _: score_derailleur_quality(raw)
    ),
    ("reliability-05", "cmp-008"): CriterionConfig(
        ("brakes_label",), lambda raw, bike: score_brakes(raw, bike.get("brakes_rotor_mm"))
    ),
    ("reliability-06", "cmp-039"): CriterionConfig(
        ("tyres_label",), lambda raw, bike: score_tyres_autonomy(raw, bike.get("category"))
    ),
    ("reliability-07", "cmp-002"): CriterionConfig(
        ("batterie_cap_wh",), lambda raw, _: score_battery_capacity(raw)
    ),
    ("reliability-08", "cmp-006"): CriterionConfig(
        ("battery_certified_ul",), lambda raw, _: score_boolean(raw), label=lambda raw, _: bool_formatter(raw)
    ),
    ("reliability-09", "cmp-037"): CriterionConfig(
        ("suspension_avant",), lambda raw, _: score_suspension(raw, allow_none_high=True)
    ),
    ("reliability-10", "cmp-036"): CriterionConfig(
        ("suspension_arriere",), lambda raw, _: score_suspension(raw, allow_none_high=True)
    ),
}


def build_tables() -> Dict[str, Any]:
    bikes = load_json(ROOT / "public" / "bikes.json")["bikes"]
    filters = load_json(ROOT / "scoring" / "filter_definitions.json")

    tables: Dict[str, Dict[str, Any]] = {}

    for filter_block in filters.values():
        for criterion in filter_block.get("criteria", []):
            key = (criterion["id"], criterion["category"])
            config = CRITERIA_CONFIG.get(key)
            if config is None:
                continue

            entry_key = f"{criterion['id']}__{criterion['category']}"
            values: Dict[str, float] = {}

            for bike in bikes:
                raw_value = collect_raw_value(bike, config.fields)
                score = config.score(raw_value, bike)
                if score is None:
                    continue
                label_func = config.label or (lambda raw, _: default_formatter(raw))
                label = label_func(raw_value, bike)
                if label is None:
                    continue
                values[label] = round(float(score), 2)

            if not values:
                continue

            ordered_values = {
                key: values[key]
                for key in sorted(values.keys(), key=lambda item: normalize_label(item))
            }

            tables[entry_key] = {
                "id": criterion["id"],
                "filterId": criterion["id"],
                "componentId": criterion["category"],
                "category": criterion["qualityCategory"],
                "values": ordered_values,
            }

    return {"tables": tables}


def main() -> None:
    tables = build_tables()
    target = ROOT / "scoring" / "component_quality.json"
    target.write_text(json.dumps(tables, indent=2, ensure_ascii=False) + "\n", encoding="utf-8")


if __name__ == "__main__":
    main()
