"""Populate bike datasets with derived performance scores."""
from __future__ import annotations

import json
from pathlib import Path
from typing import Any, Dict

ROOT = Path(__file__).resolve().parents[1]
DATA_PATH = ROOT / "public" / "bikes.json"

import sys

sys.path.append(str(ROOT))

from scoring.filters import (
    compute_offroad_performance_score,
    compute_road_performance_score,
    compute_utility_comfort_score,
)

CATEGORY_WEIGHT = {
    "urbain": 3.0,
    "polyvalent": 3.5,
    "pliant": 4.0,
    "montagne": 2.5,
    "cargo": 1.5,
    "route": 4.5,
}

CATEGORY_AERO = {
    "urbain": 3.0,
    "polyvalent": 3.5,
    "pliant": 3.5,
    "montagne": 2.0,
    "cargo": 1.0,
    "route": 5.0,
}

CATEGORY_GEOMETRY = {
    "urbain": 2.5,
    "polyvalent": 3.0,
    "pliant": 2.5,
    "montagne": 4.0,
    "cargo": 2.0,
    "route": 3.0,
}

CATEGORY_ERGONOMICS = {
    "urbain": 4.0,
    "polyvalent": 3.5,
    "pliant": 3.5,
    "montagne": 2.0,
    "cargo": 4.5,
    "route": 2.5,
}

CATEGORY_LIGHTING = {
    "urbain": 4.0,
    "polyvalent": 3.0,
    "pliant": 3.0,
    "montagne": 2.5,
    "cargo": 4.5,
    "route": 2.0,
}

CATEGORY_FENDERS = {
    "urbain": 5.0,
    "polyvalent": 4.0,
    "pliant": 3.5,
    "montagne": 1.5,
    "cargo": 4.5,
    "route": 1.0,
}

CATEGORY_RACK = {
    "urbain": 2.5,
    "polyvalent": 2.0,
    "pliant": 2.0,
    "montagne": 0.5,
    "cargo": 5.0,
    "route": 0.5,
}

CATEGORY_COMFORT = {
    "urbain": 4.0,
    "polyvalent": 3.5,
    "pliant": 3.0,
    "montagne": 2.5,
    "cargo": 4.5,
    "route": 2.0,
}

MOTOR_BRAND_SOFTWARE = {
    1: 2.5,
    2: 3.5,
    3: 4.5,
}

MOTOR_BRAND_RESPONSIVENESS = {
    1: 2.5,
    2: 3.5,
    3: 4.5,
}

POWER_TO_TORQUE = {
    1: 2.5,
    2: 3.5,
    3: 4.5,
}

THROTTLE_DEFAULT = 0.0


def build_specs(bike: Dict[str, Any]) -> Dict[str, Any]:
    specs: Dict[str, Any] = {}

    if "moteur_type" in bike:
        specs["motor_type"] = bike["moteur_type"]
    if "puissance" in bike:
        specs["motor_power"] = bike["puissance"]
        specs["motor_torque"] = POWER_TO_TORQUE.get(bike["puissance"])
    if "derailleur" in bike:
        specs["transmission"] = bike["derailleur"]
    if "pneus" in bike:
        specs["tyres_road"] = bike["pneus"]
        specs["tyres_offroad"] = bike["pneus"]
    if "cadre_susp" in bike:
        specs["suspension"] = bike["cadre_susp"]
        specs["mechanical_comfort"] = max(1.0, min(5.0, float(bike["cadre_susp"]) + 1))
    if "freins" in bike:
        specs["brakes"] = bike["freins"]

    brand_rating = bike.get("moteur_marque")
    if brand_rating is not None:
        specs["software"] = MOTOR_BRAND_SOFTWARE.get(brand_rating)
        specs["motor_responsiveness"] = MOTOR_BRAND_RESPONSIVENESS.get(brand_rating)

    category = bike.get("category")
    if category:
        if category in CATEGORY_WEIGHT:
            specs["weight"] = CATEGORY_WEIGHT[category]
        if category in CATEGORY_AERO:
            specs["aerodynamics"] = CATEGORY_AERO[category]
        if category in CATEGORY_GEOMETRY:
            specs["geometry"] = CATEGORY_GEOMETRY[category]
        if category in CATEGORY_ERGONOMICS:
            specs["ergonomics"] = CATEGORY_ERGONOMICS[category]
        if category in CATEGORY_LIGHTING:
            specs["lighting"] = CATEGORY_LIGHTING[category]
        if category in CATEGORY_FENDERS:
            specs["fenders"] = CATEGORY_FENDERS[category]
        if category in CATEGORY_RACK:
            specs["rack"] = CATEGORY_RACK[category]
        if category in CATEGORY_COMFORT:
            specs["mechanical_comfort"] = max(
                specs.get("mechanical_comfort", 0.0), CATEGORY_COMFORT[category]
            )

    specs["throttle"] = THROTTLE_DEFAULT

    return specs


def main() -> None:
    bikes = json.loads(DATA_PATH.read_text())
    for bike in bikes:
        specs = build_specs(bike)

        road = compute_road_performance_score(specs)
        if road.normalized_score is not None:
            bike["roadPerformanceScore"] = road.normalized_score

        offroad = compute_offroad_performance_score(specs)
        if offroad.normalized_score is not None:
            bike["offroadPerformanceScore"] = offroad.normalized_score

        utility = compute_utility_comfort_score(specs)
        if utility.normalized_score is not None:
            bike["utilityComfortScore"] = utility.normalized_score

    DATA_PATH.write_text(json.dumps(bikes, indent=2, ensure_ascii=False) + "\n")


if __name__ == "__main__":
    main()
