"""Weighted scoring filters for electric bike evaluations.

This module implements the road performance, off-road performance and
utility/confort filters defined for ebikereviewcanada.com.  Each filter
consumes a dictionary describing a bike specification (numerical values,
labels or booleans) and returns a :class:`FilterResult` capturing the raw
weighted score together with the normalized score on a 1–10 scale.
"""
from __future__ import annotations

import json
from dataclasses import dataclass
from importlib import resources
from typing import Any, Dict, Iterable, Mapping, Optional, Sequence, Tuple
import re
import unicodedata


@dataclass(frozen=True)
class FilterResult:
    """Result of a weighted filter evaluation.

    Attributes
    ----------
    raw_score:
        Sum of the weighted sub-category scores (0+).
    normalized_score:
        Score rescaled on a 1–10 range. ``None`` when no sub-category could
        be evaluated.
    max_raw_score:
        Maximum reachable raw score given the available sub-categories.
    breakdown:
        Mapping of the sub-category identifier to the resolved score (0–5 or
        0–2 for throttles). Missing categories are omitted from the mapping.
    """

    raw_score: float
    normalized_score: Optional[float]
    max_raw_score: float
    breakdown: Dict[str, float]


@dataclass(frozen=True)
class _SubCategory:
    """Definition of a weighted sub-category used for a filter."""

    name: str
    weight: float
    category: str
    keys: Tuple[str, ...]
    max_score: Optional[float] = None

    def resolve(self, specs: Mapping[str, Any]) -> Optional[float]:
        raw = _get_first(specs, self.keys)
        if raw is None:
            return None
        return _lookup_component_score(self.category, raw, self.max_score)


def compute_road_performance_score(specs: Mapping[str, Any]) -> FilterResult:
    """Compute the on-road performance score for an electric bike.

    Parameters
    ----------
    specs:
        Dictionary describing the bike. Keys can be either direct numerical
        sub-scores (0–5) or richer descriptors (strings, booleans). The
        function is resilient to missing values and skips unknown
        sub-categories when needed.
    """

    return _compute_filter(specs, ROAD_PERFORMANCE_SUBCATEGORIES)


def compute_offroad_performance_score(specs: Mapping[str, Any]) -> FilterResult:
    """Compute the off-road performance score for an electric bike."""

    return _compute_filter(specs, OFFROAD_PERFORMANCE_SUBCATEGORIES)


def compute_utility_comfort_score(specs: Mapping[str, Any]) -> FilterResult:
    """Compute the utility & comfort score for an electric bike."""

    return _compute_filter(specs, UTILITY_COMFORT_SUBCATEGORIES)


# ---------------------------------------------------------------------------
# Core evaluation helpers
# ---------------------------------------------------------------------------

def _compute_filter(specs: Mapping[str, Any], sub_categories: Sequence[_SubCategory]) -> FilterResult:
    breakdown: Dict[str, float] = {}
    raw_score = 0.0
    max_raw = 0.0

    for sub in sub_categories:
        max_score = sub.max_score if sub.max_score is not None else _COMPONENT_MAX_SCORES.get(sub.category, 10.0)
        if max_score <= 0:
            continue
        value = sub.resolve(specs)
        if value is None:
            continue
        coerced = float(_clamp(float(value), 0.0, max_score))
        breakdown[sub.name] = coerced
        raw_score += coerced * sub.weight
        max_raw += max_score * sub.weight

    normalized = None
    if max_raw > 0:
        normalized = 1 + (raw_score / max_raw) * 9
        normalized = round(normalized, 2)

    return FilterResult(raw_score=round(raw_score, 2), normalized_score=normalized, max_raw_score=max_raw, breakdown=breakdown)


def _normalize_label(value: str) -> str:
    normalized = unicodedata.normalize("NFKD", value)
    normalized = normalized.encode("ascii", "ignore").decode("ascii")
    normalized = normalized.lower()
    normalized = re.sub(r"[^a-z0-9]+", " ", normalized)
    return normalized.strip()


def _clamp(value: float, min_value: float, max_value: float) -> float:
    return max(min_value, min(max_value, value))


def _load_component_tables() -> Tuple[Dict[str, Dict[str, float]], Dict[str, float]]:
    """Load the component quality tables shipped with the package."""

    try:
        data_path = resources.files(__package__).joinpath("component_quality.json")
    except AttributeError:
        # ``resources.files`` is not available on Python < 3.9. Fallback to
        # ``open_text`` which works on older versions.
        with resources.open_text(__package__, "component_quality.json", encoding="utf-8") as fh:
            raw_tables = json.load(fh)
    else:
        with data_path.open("r", encoding="utf-8") as fh:
            raw_tables = json.load(fh)

    tables: Dict[str, Dict[str, float]] = {}
    max_scores: Dict[str, float] = {}

    def _record(category: Optional[str], mapping: Mapping[str, Any]) -> None:
        if not category:
            return
        normalized_mapping: Dict[str, float] = {}
        max_value = 0.0
        for label, score in mapping.items():
            normalized_label = _normalize_label(str(label))
            if not normalized_label:
                continue
            try:
                numeric_score = float(score)
            except (TypeError, ValueError):
                continue
            normalized_mapping[normalized_label] = numeric_score
            max_value = max(max_value, numeric_score)
        if normalized_mapping:
            tables[category] = normalized_mapping
            max_scores[category] = max_value if max_value > 0 else 10.0

    if isinstance(raw_tables, Mapping) and "tables" in raw_tables:
        container = raw_tables["tables"]
        if isinstance(container, Mapping):
            entries = container.values()
        elif isinstance(container, Sequence):
            entries = container
        else:
            entries = []

        for entry in entries:
            if not isinstance(entry, Mapping):
                continue
            category = entry.get("category")
            values = entry.get("values")
            if isinstance(values, Mapping):
                _record(str(category) if category else None, values)
        if tables:
            return tables

    if isinstance(raw_tables, Mapping):
        for category, mapping in raw_tables.items():
            if isinstance(mapping, Mapping):
                _record(str(category), mapping)
    return tables, max_scores


_COMPONENT_SCORE_TABLES, _COMPONENT_MAX_SCORES = _load_component_tables()


def _lookup_component_score(category: str, raw: Any, max_score: Optional[float] = None) -> Optional[float]:
    """Resolve a component rating using the external quality tables."""

    if raw is None:
        return None
    table = _COMPONENT_SCORE_TABLES.get(category)
    if not table:
        return None

    limit = max_score if max_score is not None else _COMPONENT_MAX_SCORES.get(category, 10.0)
    if limit <= 0:
        return None

    if isinstance(raw, (list, tuple, set)):
        for item in raw:
            score = _lookup_component_score(category, item, max_score=limit)
            if score is not None:
                return score
        return None

    if isinstance(raw, bool):
        label = "true" if raw else "false"
    else:
        label = _normalize_label(str(raw))

    if not label:
        return None

    score = table.get(label)
    if score is None and isinstance(raw, (str, int, float)):
        numbers = re.findall(r"\d+(?:\.\d+)?", str(raw))
        for candidate in numbers:
            alt = _normalize_label(candidate)
            score = table.get(alt)
            if score is not None:
                break
    if score is None:
        return None
    return float(_clamp(score, 0.0, limit))


def _get_first(specs: Mapping[str, Any], keys: Iterable[str]) -> Any:
    for key in keys:
        if key in specs:
            return specs[key]
    return None




MOTOR_TYPE_KEYS: Tuple[str, ...] = (
    "road_motor_type",
    "motor_type",
    "motorType",
    "motor_type_score",
    "onroad_motor_type",
    "offroad_motor_type",
    "assist_type",
)

MOTOR_POWER_KEYS: Tuple[str, ...] = (
    "road_motor_power",
    "motor_power",
    "motorPower",
    "power_nominal",
    "puissance_nominale",
    "motor_power_w",
    "motor_power_score",
)

MOTOR_TORQUE_KEYS: Tuple[str, ...] = (
    "road_motor_torque",
    "offroad_motor_torque",
    "motor_torque",
    "torque",
    "torque_nm",
    "motor_torque_nm",
    "motor_torque_score",
)

ROAD_TRANSMISSION_KEYS: Tuple[str, ...] = (
    "derailleur_speeds",
    "cassette_speeds",
    "speeds",
    "vitesses",
    "road_transmission",
    "transmission",
)

OFFROAD_TRANSMISSION_KEYS: Tuple[str, ...] = (
    "derailleur_label",
    "derailleur_model",
    "offroad_transmission",
    "transmission",
    "drivetrain",
    "transmission_label",
    "rear_derailleur",
)

ROAD_TYRE_KEYS: Tuple[str, ...] = (
    "road_tyres",
    "tyres_road",
    "pneus_road",
    "road_tires",
    "tyres_label",
    "tyres",
)

OFFROAD_TYRE_KEYS: Tuple[str, ...] = (
    "offroad_tyres",
    "mtb_tyres",
    "tyres_mtb",
    "pneus_vtt",
    "tyres",
    "tyres_label",
)

WEIGHT_KEYS: Tuple[str, ...] = (
    "weight",
    "poids",
    "bike_weight",
    "poids_kg",
    "weight_kg",
)

AERODYNAMICS_KEYS: Tuple[str, ...] = (
    "aerodynamics",
    "frame_aero",
    "road_aero",
    "cadre_aero",
    "geometry_aero",
    "category",
    "rawCategory",
    "bike_category",
)

SOFTWARE_KEYS: Tuple[str, ...] = (
    "display",
    "screen",
    "console",
    "ecran",
    "assistance_logiciel",
    "assistance_logicielle",
    "assistance_software",
    "software_support",
    "motor_assistance",
    "assist_levels",
)

THROTTLE_KEYS: Tuple[str, ...] = (
    "throttle",
    "has_throttle",
    "accelerateur",
    "gas_handle",
)

FRONT_SUSPENSION_KEYS: Tuple[str, ...] = (
    "suspension_avant",
    "front_suspension",
    "fork",
    "fork_model",
    "fork_travel",
    "suspension",
    "suspension_type",
)

REAR_SUSPENSION_KEYS: Tuple[str, ...] = (
    "suspension_arriere",
    "rear_suspension",
    "frame_suspension",
    "rear_shock",
    "rear_travel",
    "suspension",
)

BRAKES_KEYS: Tuple[str, ...] = (
    "brakes",
    "brake_system",
    "freins",
    "brakes_model",
    "brakes_label",
)

BRAKES_ROTOR_KEYS: Tuple[str, ...] = (
    "brakes_rotor_mm",
    "brake_rotor_mm",
    "rotor_front_mm",
    "rotor_size",
    "rotor_front",
    "front_rotor",
    "brakes_rotor",
)

POSITION_KEYS: Tuple[str, ...] = (
    "stem_adjustable",
    "ergonomics",
    "rider_position",
    "posture",
    "position",
)

LIGHTS_FRONT_KEYS: Tuple[str, ...] = (
    "lights_front",
    "lighting_front",
    "eclairage_avant",
    "lumiere_avant",
)

LIGHTS_REAR_KEYS: Tuple[str, ...] = (
    "lights_rear",
    "lighting_rear",
    "eclairage_arriere",
    "lumiere_arriere",
)

FENDERS_KEYS: Tuple[str, ...] = (
    "fenders",
    "garde_boue",
    "mudguards",
    "fenders_front",
    "fenders_rear",
)

RACK_FRONT_KEYS: Tuple[str, ...] = (
    "rack_front",
    "front_rack",
    "porte_bagages_avant",
    "rack_avant",
)

RACK_REAR_KEYS: Tuple[str, ...] = (
    "rack_rear",
    "rear_rack",
    "porte_bagages",
    "porte_bagages_arriere",
    "rack",
)


ROAD_PERFORMANCE_SUBCATEGORIES: Sequence[_SubCategory] = (
    _SubCategory("motor_type", 2.0, "road_performance-01", MOTOR_TYPE_KEYS),
    _SubCategory("motor_power", 1.0, "road_performance-02", MOTOR_POWER_KEYS),
    _SubCategory("motor_torque", 2.5, "road_performance-03", MOTOR_TORQUE_KEYS),
    _SubCategory("transmission", 2.0, "road_performance-04", ROAD_TRANSMISSION_KEYS),
    _SubCategory("tyres_road", 3.0, "road_performance-05", ROAD_TYRE_KEYS),
    _SubCategory("weight", 3.0, "road_performance-06", WEIGHT_KEYS),
    _SubCategory("aerodynamics", 1.5, "road_performance-07", AERODYNAMICS_KEYS),
    _SubCategory("software", 3.0, "road_performance-08", SOFTWARE_KEYS),
    _SubCategory("throttle", 3.0, "road_performance-09", THROTTLE_KEYS),
)


OFFROAD_PERFORMANCE_SUBCATEGORIES: Sequence[_SubCategory] = (
    _SubCategory("motor_type", 1.5, "offroad_performance-01", MOTOR_TYPE_KEYS),
    _SubCategory("motor_torque", 2.5, "offroad_performance-02", MOTOR_TORQUE_KEYS),
    _SubCategory("transmission_quality", 1.5, "offroad_performance-03", OFFROAD_TRANSMISSION_KEYS),
    _SubCategory("front_suspension", 1.5, "offroad_performance-04", FRONT_SUSPENSION_KEYS),
    _SubCategory("rear_suspension", 1.5, "offroad_performance-05", REAR_SUSPENSION_KEYS),
    _SubCategory("brakes", 2.0, "offroad_performance-06", BRAKES_KEYS),
    _SubCategory("brakes_rotor", 2.0, "offroad_performance-10", BRAKES_ROTOR_KEYS),
    _SubCategory("tyres_offroad", 1.5, "offroad_performance-07", OFFROAD_TYRE_KEYS),
    _SubCategory("weight", 2.0, "offroad_performance-08", WEIGHT_KEYS),
    _SubCategory("throttle", 1.0, "offroad_performance-09", THROTTLE_KEYS),
)


UTILITY_COMFORT_SUBCATEGORIES: Sequence[_SubCategory] = (
    _SubCategory("ergonomics", 2.0, "utility_comfort-01", POSITION_KEYS),
    _SubCategory("lighting_front", 2.0, "utility_comfort-02", LIGHTS_FRONT_KEYS),
    _SubCategory("lighting_rear", 1.0, "utility_comfort-06", LIGHTS_REAR_KEYS),
    _SubCategory("fenders", 2.0, "utility_comfort-03", FENDERS_KEYS),
    _SubCategory("rack_front", 1.0, "utility_comfort-07", RACK_FRONT_KEYS),
    _SubCategory("rack_rear", 2.0, "utility_comfort-04", RACK_REAR_KEYS),
    _SubCategory("throttle", 2.0, "utility_comfort-05", THROTTLE_KEYS),
)
