const GLOBAL_METRIC_DEFINITIONS = {
  fiabiliteScore: { label: "Fiabilité" },
  autonomieScore: { label: "Autonomie" },
  utilityComfortScore: { label: "Utilité & confort" },
  roadPerformanceScore: { label: "Efficacité sur l’asphalte" },
  offroadPerformanceScore: { label: "Maîtrise en terrain naturel" },
  privatePerformanceScore: { label: "Capacité d’exploration" },
  winterPerformanceScore: { label: "Endurance en conditions extrêmes" },
  globalScore: { label: "Score sélection" },
};

const METRIC_COLOR_MAP = {
  fiabiliteScore: "#0ea5e9",
  autonomieScore: "#22c55e",
  utilityComfortScore: "#f97316",
  roadPerformanceScore: "#6366f1",
  offroadPerformanceScore: "#14b8a6",
  privatePerformanceScore: "#8b5cf6",
  winterPerformanceScore: "#38bdf8",
};

const METRIC_COLOR_FALLBACKS = [
  "#0ea5e9",
  "#22c55e",
  "#6366f1",
  "#f97316",
  "#14b8a6",
  "#8b5cf6",
  "#38bdf8",
  "#f59e0b",
];

const DEFAULT_GLOBAL_METRIC_KEYS = [
  "fiabiliteScore",
  "autonomieScore",
  "utilityComfortScore",
  "roadPerformanceScore",
];

const USE_CASES = [
  {
    id: "urban-leisure",
    label: "Balade en ville / trajets urbains",
    metrics: ["roadPerformanceScore", "utilityComfortScore"],
    weightKey: "balade_urbaine",
    image: "balade-ville.png",
    alt: "Cycliste urbain se déplaçant en ville.",
  },
  {
    id: "commute",
    label: "Vélotaf (trajets domicile-travail)",
    metrics: ["roadPerformanceScore", "utilityComfortScore", "autonomieScore"],
    weightKey: "deplacements_quotidiens",
    image: "velotaf.png",
    alt: "Cycliste de dos se rendant au travail avec un sac à dos.",
  },
  {
    id: "family",
    label: "Loisirs / balade familiale",
    metrics: ["roadPerformanceScore", "utilityComfortScore"],
    weightKey: "loisirs_familiaux",
    image: "loisirs-famille.png",
    alt: "Famille à vélo pendant une balade.",
  },
  {
    id: "mtb-sport",
    label: "Vélo de montagne sportif",
    metrics: ["offroadPerformanceScore", "autonomieScore"],
    weightKey: "montagne_sportif",
    image: "vtt-sportif.png",
    alt: "Cycliste pratiquant le VTT sur un sentier.",
  },
  {
    id: "private-grounds",
    label: "Chasse / terrain privé",
    metrics: ["privatePerformanceScore", "autonomieScore"],
    weightKey: "chasse_prive",
    image: "chasse-prive.png",
    alt: "Cycliste équipé pour la chasse sur un terrain privé.",
  },
  {
    id: "long-adventure",
    label: "Aventure longue distance (bikepacking / voyage)",
    metrics: ["autonomieScore", "roadPerformanceScore", "utilityComfortScore"],
    weightKey: "aventure_longue_distance",
    image: "aventure-longue-distance.png",
    alt: "Cycliste chargé pour un voyage longue distance.",
  },
  {
    id: "utility",
    label: "Utilitaire (courses, transport de charge)",
    metrics: ["utilityComfortScore", "privatePerformanceScore"],
    weightKey: "utilitaire",
    image: "test.png",
    alt: "Cycliste transportant des courses sur son vélo.",
  },
  {
    id: "winter",
    label: "Hiver / neige / conditions extrêmes",
    metrics: ["winterPerformanceScore", "autonomieScore"],
    weightKey: "hiver_conditions_extremes",
    image: "hiver-neige.png",
    alt: "Cycliste en fatbike dans la neige.",
  },
];

const USE_CASE_MAP = new Map(USE_CASES.map(useCase => [useCase.id, useCase]));

const ALL_METRIC_KEYS = Object.keys(GLOBAL_METRIC_DEFINITIONS);

const BIKE_IMAGE_MAP = new Map([
  ["biktrix-juggernaut-ultra-fs-pro-3", "model/Juggernaut_Ultra_FSPro_3.jpg"],
  ["biktrix-juggernaut-xd", "model/Juggernaut_XD.jpg"],
  ["biktrix-stunner-x-6", "model/stunner_x_6.jpg"],
  ["devinci-e-milano", "model/e_milano.jpg"],
  ["devinci-e-spartan-lite", "model/e_spartan.jpg"],
  ["devinci-e-troy", "model/e_troy.jpg"],
  ["envo_d50", "model/d50.jpg"],
  ["envo_flex_overland", "model/envo_flex_overland.jpg"],
  ["envo_st50", "model/st50.jpg"],
  ["katch-horizon", "model/horizon.jpg"],
  ["katch-horizon-vee", "model/horizonVee.jpg"],
  ["katch-horizon-mini", "model/mini-gris.jpg"],
  ["katch-alpha-sram-eagle", "model/alpha_sram.jpg"],
  ["norco-fluid-vlt", "model/fluid_vlt.jpg"],
  ["norco-range-vlt", "model/range_vlt_c1.jpg"],
  ["norco-sight-vlt", "model/sight_vlt.jpg"],
  ["ohm-cruise-3", "model/cruise_3.jpg"],
  ["ohm-journey-pro-ep8", "model/Journey_pro.jpg"],
  ["ohm-quest", "model/quest3.jpg"],
  ["opus-connect-lrt", "model/connect_lrt.jpg"],
  ["opus-ebigcity-lrt", "model/e_big_city.jpg"],
  ["opus-wknd-lrt", "model/wknd.jpg"],
  ["rockymountain-altitude-powerplay", "model/altitude_powerplay_c70.jpg"],
  ["rockymountain-blizzard-powerplay", "model/blizzard.jpg"],
  ["rockymountain-growler-powerplay", "model/Growler_Powerplay.jpg"],
  ["rockymountain-instinct-powerplay", "model/instinct_alloy_30.jpg"],
  ["rockymountain-instinct-powerplay-sl", "model/powerplay.jpg"],
  ["surface604_quad_2021", "model/quad.jpg"],
  ["surface604_shred_2022", "model/SHRED.jpg"],
  ["surface604_twist_2022", "model/twist.jpg"],
  ["teovelo-ecargo", "model/ecargo.jpg"],
  ["teovelo-emini", "model/emini.jpg"],
  ["teovelo-esuv-560", "model/esuv560.jpg"],
  ["velec-a2", "model/A2.jpg"],
  ["velec-citi-350", "model/citi_350.jpg"],
  ["velec-r48i", "model/r48i.jpg"],
  ["voltbike_yukon_750_limited", "model/yukon_750.jpg"],
  ["voltbike_yukon_elite_step_thru", "model/Yukon_Step_Thru.jpg"],
  ["voltbike_yukon_v2", "model/yukon_v2.jpg"],
]);

const state = {
  data: [],
  query: "",
  category: "toutes",
  sortKey: "globalScore",
  sortDir: "desc",
  showOnlyScored: false,
  autonomyThreshold: 0,
  selectedUseCases: new Set(),
  selectedGlobalMetrics: [...DEFAULT_GLOBAL_METRIC_KEYS],
};

const DESC_SORT_KEYS = new Set([
  "globalScore",
  "fiabiliteScore",
  "autonomieScore",
  "utilityComfortScore",
  "roadPerformanceScore",
  "offroadPerformanceScore",
  "privatePerformanceScore",
  "winterPerformanceScore",
]);

let initialData = [];

const tableBody = document.querySelector("#table-body");
const resultsCount = document.querySelector("#results-count");
const searchInput = document.querySelector("#search");
const categorySelect = document.querySelector("#category");
const onlyScoredCheckbox = document.querySelector("#only-scored");
const autonomyThresholdInput = document.querySelector("#autonomy-threshold");
const autonomyThresholdValue = document.querySelector("#autonomy-threshold-value");
const useCaseGrid = document.querySelector("#use-case-grid");
const useCaseSummary = document.querySelector("#use-case-summary");
const sortButtons = Array.from(document.querySelectorAll(".sort"));
const sortableHeaderCells = Array.from(document.querySelectorAll("thead [data-score-key]"));
const resetButton = document.querySelector("#reset");
const weightUpdateIndicator = document.querySelector("#weight-update-indicator");

if (weightUpdateIndicator) {
  weightUpdateIndicator.dataset.visible = "false";
}
const scorePopover = document.querySelector("#score-popover");
const scorePopoverTitle = document.querySelector("#score-popover-title");
const scorePopoverBody = document.querySelector("#score-popover-body");

let useCaseCheckboxes = [];

const brandMetadata = new Map();
let componentIdMap = null;
let componentQualityTables = null;
let componentQualityIndex = null;
let componentKeyIndex = null;
let filterDefinitions = {};
let filterMetadata = {};
let useCaseWeightsConfig = null;
let cachedMetricWeightMap = null;
let previousMetricWeights = null;
let weightIndicatorTimeoutId = null;

const MULTIPLIER_DISPLAY_THRESHOLD = 0.01;

const REQUIRED_FILTER_KEYS = [
  "autonomy",
  "road_performance",
  "offroad_performance",
  "private_performance",
  "winter_performance",
  "utility_comfort",
  "reliability",
];

const OPTIONAL_FILTER_KEYS = new Set(["gravel_performance"]);

const SCORE_BREAKDOWN_KEYS = {
  globalScore: "globalScoreBreakdown",
  fiabiliteScore: "fiabiliteBreakdown",
  autonomieScore: "autonomieBreakdown",
  roadPerformanceScore: "roadPerformanceBreakdown",
  offroadPerformanceScore: "offroadPerformanceBreakdown",
  utilityComfortScore: "utilityComfortBreakdown",
  privatePerformanceScore: "privatePerformanceBreakdown",
  winterPerformanceScore: "winterPerformanceBreakdown",
};

const SCORE_DETAIL_RESOLVERS = {
  fiabiliteScore: row => row.fiabiliteDetails ?? null,
  globalScore: row => row.globalScoreDetails ?? null,
  autonomieScore: row => row.autonomieDetails ?? null,
  roadPerformanceScore: row => row.roadPerformanceDetails ?? null,
  offroadPerformanceScore: row => row.offroadPerformanceDetails ?? null,
  utilityComfortScore: row => row.utilityComfortDetails ?? null,
  privatePerformanceScore: row => row.privatePerformanceDetails ?? null,
  winterPerformanceScore: row => row.winterPerformanceDetails ?? null,
};

const SCORE_DATA_REQUIREMENTS = {
  globalScore: {
    files: ["public/bikes.json"],
    fields: [
      "fiabiliteScore",
      "autonomieScore",
      "roadPerformanceScore",
      "offroadPerformanceScore",
      "utilityComfortScore",
    ],
  },
  fiabiliteScore: {
    files: ["public/bikes.json", "scoring/component_quality.json"],
    fields: [
      "motor_brand / motor_label",
      "controller",
      "display",
      "derailleur_label",
      "brakes_label",
      "tyres_label",
      "batteryWh / battery_certification",
      "battery_cells",
      "weight_kg",
      "stem_adjustable",
      "suspension_avant",
      "suspension_arriere",
    ],
    note:
      "Assure-toi que les tables de qualité sont renseignées dans scoring/component_quality.json (catégories Moteur, Freins, etc.).",
  },
  autonomieScore: {
    files: ["public/bikes.json"],
    fields: [
      "batterie_cap_wh",
      "battery_cells",
      "range_extender",
      "motor_type",
      "motor_power",
      "frame_suspension",
      "tyres_label",
      "transmission",
    ],
  },
  roadPerformanceScore: {
    files: ["public/bikes.json"],
    fields: [
      "motor_type",
      "motor_power",
      "motor_torque_nm",
      "transmission",
      "tyres_label",
      "weight_kg",
      "aerodynamics",
      "assistance_logiciel",
      "throttle",
    ],
  },
  offroadPerformanceScore: {
    files: ["public/bikes.json"],
    fields: [
      "motor_type",
      "motor_torque_nm",
      "motor_responsiveness",
      "transmission",
      "suspension",
      "brakes_label",
      "tyres_label",
      "geometry",
      "weight_kg",
      "throttle",
    ],
  },
  utilityComfortScore: {
    files: ["public/bikes.json"],
    fields: [
      "ergonomics",
      "lighting",
      "fenders",
      "rack",
      "throttle",
      "mechanical_comfort",
    ],
  },
  privatePerformanceScore: {
    files: ["public/bikes.json"],
    fields: [
      "motor_type",
      "motor_torque_nm",
      "rack",
      "lighting",
      "tyres_label",
      "brakes_label",
      "weight_kg",
    ],
  },
  winterPerformanceScore: {
    files: ["public/bikes.json"],
    fields: [
      "tyres_label",
      "tyres_width",
      "motor_torque_nm",
      "battery_heating",
      "fenders",
      "lighting",
    ],
  },
};

let activeScoreCell = null;

function invalidateMetricWeightCache() {
  cachedMetricWeightMap = null;
}

function getUseCaseWeightKey(useCaseId) {
  const config = USE_CASE_MAP.get(useCaseId);
  return config?.weightKey ?? useCaseId;
}

function setUseCaseWeightConfig(config) {
  if (!config || typeof config !== "object") {
    return false;
  }

  useCaseWeightsConfig = config;
  invalidateMetricWeightCache();

  const missing = USE_CASES.filter(useCase => {
    const key = getUseCaseWeightKey(useCase.id);
    const entry = config[key];
    return !entry || typeof entry.weights !== "object";
  }).map(useCase => useCase.id);

  if (missing.length) {
    console.warn(
      `Pondérations manquantes pour les cas d'usage : ${missing.join(", ")}`
    );
  }

  return true;
}

function getUseCaseWeightProfile(useCaseId) {
  if (!useCaseId || !useCaseWeightsConfig) return null;
  const key = getUseCaseWeightKey(useCaseId);
  const entry = useCaseWeightsConfig?.[key];
  const weights = entry?.weights;
  return weights && typeof weights === "object" ? weights : null;
}

function getMetricKeysFromProfile(profile) {
  if (!profile || typeof profile !== "object") return [];
  return Object.entries(profile)
    .filter(([metric, value]) => {
      return (
        GLOBAL_METRIC_DEFINITIONS[metric] &&
        typeof value === "number" &&
        Number.isFinite(value) &&
        value > 0
      );
    })
    .map(([metric]) => metric);
}

function getActiveUseCaseWeightProfiles() {
  if (!useCaseWeightsConfig) return [];
  if (!state.selectedUseCases || state.selectedUseCases.size === 0) return [];

  const profiles = [];
  state.selectedUseCases.forEach(id => {
    const weights = getUseCaseWeightProfile(id);
    if (weights && typeof weights === "object") {
      profiles.push(weights);
    }
  });
  return profiles;
}

function getActiveMetricWeightMap() {
  if (cachedMetricWeightMap) {
    return cachedMetricWeightMap;
  }

  const result = {};
  const profiles = getActiveUseCaseWeightProfiles();

  if (!profiles.length) {
    ALL_METRIC_KEYS.forEach(key => {
      result[key] = 1;
    });
    cachedMetricWeightMap = result;
    return result;
  }

  ALL_METRIC_KEYS.forEach(key => {
    let sum = 0;
    let count = 0;
    profiles.forEach(profile => {
      const raw = profile?.[key];
      if (typeof raw === "number" && Number.isFinite(raw)) {
        sum += raw;
        count += 1;
      }
    });
    result[key] = count ? sum / count : 1;
  });

  cachedMetricWeightMap = result;
  return result;
}

function getMetricWeight(metricKey) {
  const weights = getActiveMetricWeightMap();
  const value = weights?.[metricKey];
  return typeof value === "number" && Number.isFinite(value) ? value : 1;
}

function formatRequirementMessage(scoreKey) {
  const requirements = SCORE_DATA_REQUIREMENTS[scoreKey];
  if (!requirements) return null;

  const files = Array.isArray(requirements.files) ? requirements.files : [];
  const fields = Array.isArray(requirements.fields) ? requirements.fields : [];
  const notes = [];

  if (files.length) {
    const formattedFiles = files.length === 1
      ? files[0]
      : `${files.slice(0, -1).join(", ")} et ${files.slice(-1)[0]}`;
    notes.push(`Complète les données dans ${formattedFiles}.`);
  }

  if (fields.length) {
    notes.push(`Champs requis : ${fields.join(", ")}.`);
  }

  if (requirements.note) {
    notes.push(requirements.note);
  }

  return notes.join(" ");
}

function setBrandMetadata(entries) {
  brandMetadata.clear();
  if (!Array.isArray(entries)) return;

  entries.forEach(entry => {
    const brandName = getStringValue(entry, ["brand", "name", "marque"]);
    if (!brandName) return;

    const key = brandName.toLowerCase();
    const website = getStringValue(entry, [
      "website",
      "url",
      "site",
      "link",
      "brandWebsite",
      "brand_url",
    ]);
    const legalName = getStringValue(entry, [
      "legalName",
      "company",
      "companyName",
      "raisonSociale",
    ]);
    const foundedValue = getNumericValue(entry, [
      "foundedYear",
      "founded",
      "year",
      "creationYear",
      "anneeCreation",
    ]);
    const foundedYear = normalizeFoundedYear(foundedValue);

    brandMetadata.set(key, {
      brand: brandName,
      website: website ?? null,
      foundedYear,
      legalName: legalName ?? null,
    });
  });
}

function getBrandMetadata(brand) {
  if (!brand) return null;
  return brandMetadata.get(String(brand).toLowerCase()) ?? null;
}

function applyBrandMetadata(bike) {
  if (!bike?.brand) return bike;
  const meta = getBrandMetadata(bike.brand);
  if (!meta) return bike;

  if (!bike.brandWebsite && meta.website) {
    bike.brandWebsite = meta.website;
  }
  if (!bike.brandFoundedYear && meta.foundedYear) {
    bike.brandFoundedYear = meta.foundedYear;
  }
  if (!bike.brandLegalName && meta.legalName && meta.legalName !== bike.brand) {
    bike.brandLegalName = meta.legalName;
  }
  return bike;
}

function normalizeBikeDataset(rawDataset) {
  if (Array.isArray(rawDataset)) {
    return { bikes: rawDataset, componentIds: null };
  }

  if (!rawDataset || typeof rawDataset !== "object") {
    throw new Error(
      "Format inattendu pour public/bikes.json (tableau ou objet attendu)"
    );
  }

  const { bikes, componentIds } = rawDataset;
  if (!Array.isArray(bikes)) {
    throw new Error(
      "Format inattendu pour public/bikes.json (clé \"bikes\" manquante ou invalide)"
    );
  }

  let normalizedComponentIds = null;
  if (componentIds && typeof componentIds === "object") {
    normalizedComponentIds = {};
    Object.entries(componentIds).forEach(([key, value]) => {
      if (!key) return;
      if (typeof value !== "string") return;
      normalizedComponentIds[String(key)] = value;
    });
    if (Object.keys(normalizedComponentIds).length === 0) {
      normalizedComponentIds = null;
    }
  }

  return { bikes, componentIds: normalizedComponentIds };
}

function setComponentQualityTables(rawTables) {
  componentQualityTables = null;
  componentQualityIndex = null;
  if (!rawTables || typeof rawTables !== "object") {
    return;
  }

  const registry = {};

  const registerTable = (id, category, values) => {
    if (!values || typeof values !== "object") {
      return;
    }

    const normalizedValues = {};
    let defaultScore = null;
    let maxScore = 0;

    Object.entries(values).forEach(([label, score]) => {
      const numericScore = Number(score);
      if (!Number.isFinite(numericScore)) {
        return;
      }
      if (label === "_default") {
        defaultScore = numericScore;
        if (numericScore > maxScore) {
          maxScore = numericScore;
        }
        return;
      }

      const normalizedLabel = normalizeLabel(String(label));
      if (!normalizedLabel) {
        return;
      }
      normalizedValues[normalizedLabel] = numericScore;
      if (numericScore > maxScore) {
        maxScore = numericScore;
      }
    });

    if (!Object.keys(normalizedValues).length && defaultScore == null) {
      return;
    }

    const key = typeof id === "string" && id ? id : typeof category === "string" && category ? category : null;
    if (!key) {
      return;
    }

    const entry = {
      values: normalizedValues,
      defaultScore: defaultScore,
      maxScore: maxScore > 0 ? maxScore : defaultScore ?? 0,
    };

    registry[key] = entry;

    if (typeof category === "string" && category && !registry[category]) {
      registry[category] = entry;
    }
  };

  const processEntry = (key, entry) => {
    if (!entry || typeof entry !== "object") {
      return;
    }

    if (entry.values && typeof entry.values === "object") {
      const id = typeof entry.id === "string" && entry.id ? entry.id : key;
      const category = typeof entry.category === "string" && entry.category ? entry.category : null;
      registerTable(id, category, entry.values);
      return;
    }

    const id = typeof entry.id === "string" && entry.id ? entry.id : key;
    const category = typeof entry.category === "string" && entry.category ? entry.category : null;
    registerTable(id, category, entry);
  };

  if (Array.isArray(rawTables.tables)) {
    rawTables.tables.forEach(item => processEntry(null, item));
  } else if (rawTables.tables && typeof rawTables.tables === "object") {
    Object.entries(rawTables.tables).forEach(([key, value]) => processEntry(key, value));
  } else {
    Object.entries(rawTables).forEach(([key, value]) => processEntry(key, value));
  }

  componentQualityTables = Object.keys(registry).length ? registry : null;
}

function buildComponentKeyIndex(map) {
  componentKeyIndex = null;
  if (!map || typeof map !== "object") {
    return;
  }

  const index = {};
  Object.entries(map).forEach(([key, value]) => {
    if (typeof key !== "string" || typeof value !== "string") {
      return;
    }
    if (!index[value]) {
      index[value] = [];
    }
    if (!index[value].includes(key)) {
      index[value].push(key);
    }
  });

  componentKeyIndex = Object.keys(index).length ? index : null;
}

function getComponentKeysForId(componentId) {
  if (!componentId || !componentKeyIndex) return null;
  const keys = componentKeyIndex[componentId];
  if (!keys) return null;
  return Array.isArray(keys) ? keys : null;
}

function collectComponentValuesById(componentId, primary, secondary) {
  const keys = getComponentKeysForId(componentId);
  if (!keys || !keys.length) return null;
  return collectValuesFromSources(primary, secondary, keys);
}

function createComponentQualityResolver(qualityKey, componentId, maxScore) {
  if (!qualityKey) return null;
  const limit = Number.isFinite(maxScore) && maxScore > 0 ? maxScore : 10;
  return (source, originalSource) => {
    if (!componentQualityTables) return null;
    const values = collectComponentValuesById(componentId ?? qualityKey, originalSource, source);
    if (!Array.isArray(values) || values.length === 0) {
      const fallback = getComponentTableDefaultScore(qualityKey);
      return fallback != null ? clamp(fallback, 0, limit) : null;
    }

    const resolved = lookupComponentScore(qualityKey, values, limit);
    if (resolved != null) {
      return resolved;
    }

    const fallback = getComponentTableDefaultScore(qualityKey);
    return fallback != null ? clamp(fallback, 0, limit) : null;
  };
}

function setFilterDefinitions(rawDefinitions) {
  filterDefinitions = {};
  filterMetadata = {};
  if (!rawDefinitions || typeof rawDefinitions !== "object") {
    return false;
  }

  let totalEntries = 0;

  const extractEntries = definition => {
    if (Array.isArray(definition)) {
      return definition.filter(item => item && typeof item === "object");
    }

    if (!definition || typeof definition !== "object") {
      return null;
    }

    const candidateKeys = ["criteria", "criteres", "criterias", "items", "points"];
    for (const key of candidateKeys) {
      const value = definition[key];
      if (Array.isArray(value)) {
        return value.filter(item => item && typeof item === "object");
      }
      if (value && typeof value === "object") {
        const entries = Object.values(value).filter(item => item && typeof item === "object");
        if (entries.length) {
          return entries;
        }
      }
    }

    const directValues = Object.entries(definition)
      .filter(([key]) => key !== "meta")
      .map(([, value]) => value)
      .filter(item => item && typeof item === "object" && !Array.isArray(item));
    if (directValues.length) {
      return directValues;
    }

    return null;
  };

  Object.entries(rawDefinitions).forEach(([filterKey, definition]) => {
    const entries = extractEntries(definition);
    if (!entries || !entries.length) {
      console.warn(`Configuration invalide pour le filtre "${filterKey}" (aucun critère détecté).`);
      return;
    }

    if (definition && typeof definition === "object" && definition.meta && typeof definition.meta === "object") {
      const meta = definition.meta;
      const normalizedMeta = {};
      if (meta.title != null) normalizedMeta.title = String(meta.title);
      if (meta.description != null) normalizedMeta.description = String(meta.description);
      Object.keys(normalizedMeta).length && (filterMetadata[filterKey] = normalizedMeta);
    }

    const normalizedEntries = entries
      .map(entry => {
        if (!entry || typeof entry !== "object") {
          return null;
        }

        const weightValue = Number(entry.weight);
        const weight = Number.isFinite(weightValue) && weightValue > 0 ? weightValue : 1;

        const rawId = typeof entry.id === "string" ? entry.id : null;
        const key = entry.key != null ? String(entry.key) : rawId ?? "";
        const label = entry.label != null ? String(entry.label) : key || (rawId ?? "");

        const normalizedEntry = {
          key,
          label,
          weight,
        };

        if (rawId) {
          normalizedEntry.id = rawId;
        }

        const rawCategory =
          typeof entry.category === "string" && entry.category ? entry.category : null;
        const explicitComponentId =
          typeof entry.componentId === "string" && entry.componentId
            ? entry.componentId
            : typeof entry.component_id === "string" && entry.component_id
            ? entry.component_id
            : null;

        const componentId = explicitComponentId ?? rawCategory ?? null;
        if (componentId) {
          normalizedEntry.componentId = componentId;
          normalizedEntry.category = componentId;
        }

        const maxScoreValue = Number(entry.maxScore);
        const maxScore = Number.isFinite(maxScoreValue) && maxScoreValue > 0 ? maxScoreValue : 10;
        normalizedEntry.maxScore = maxScore;

        if (Object.prototype.hasOwnProperty.call(entry, "fallback")) {
          normalizedEntry.fallback = entry.fallback;
        }

        const qualityKey =
          rawId ??
          (typeof entry.qualityCategory === "string" && entry.qualityCategory
            ? entry.qualityCategory
            : typeof entry.quality_category === "string" && entry.quality_category
            ? entry.quality_category
            : componentId);

        if (qualityKey) {
          normalizedEntry.qualityKey = qualityKey;
          const resolver = createComponentQualityResolver(qualityKey, componentId, maxScore);
          if (resolver) {
            normalizedEntry.resolver = resolver;
          }
        }

        return normalizedEntry;
      })
      .filter(Boolean);

    if (normalizedEntries.length) {
      filterDefinitions[filterKey] = normalizedEntries;
      totalEntries += normalizedEntries.length;
    }
  });

  return totalEntries > 0;
}

function getFilterEntries(filterKey) {
  if (!filterDefinitions || typeof filterDefinitions !== "object") return null;
  const entries = filterDefinitions[filterKey];
  return Array.isArray(entries) ? entries : null;
}

function hasFilterEntries(filterKey) {
  const entries = getFilterEntries(filterKey);
  return Array.isArray(entries) && entries.length > 0;
}

const CATEGORY_SET = new Set(["urbain", "montagne", "polyvalent", "cargo", "pliant", "route"]);
const NUMERIC_FIELDS = {
  totalPoints: [],
  moteur_marque: [],
  moteur_type: [],
  puissance: [],
  controleur: [],
  afficheur: [],
  derailleur: [],
  pneus: [],
  batterie_certif: [],
  batterie_cap_ah: [],
  batterie_cap_wh: ["batteryWh", "battery_wh", "batteryCapacityWh", "batterie_cap_wh"],
  battery_capacity_score: [
    "battery_capacity_score",
    "batteryCapacityScore",
    "autonomie_batterie_capacite",
    "autonomie_batterie_score",
  ],
  cellules_type: [],
  battery_cells_score: [
    "battery_cells_score",
    "batteryCellsScore",
    "autonomie_batterie_cellules",
  ],
  freins: [],
  cadre_susp: [],
  frame_efficiency_score: [
    "frameEfficiencyScore",
    "autonomie_cadre_score",
    "cadre_efficiency_score",
  ],
  batterie_continu: ["batterieContinu"],
  bonus_add_on: ["bonusAddon", "bonus_addon", "bonus_add-on"],
  range_extender_score: [
    "rangeExtenderScore",
    "range_extender_score",
    "autonomie_add_on",
    "battery_add_on",
  ],
  moteur_assistance: ["moteurAssistance"],
  motor_type_score: ["motorTypeScore", "motor_type_score", "autonomie_motor_type"],
  poids: ["weight"],
  motor_power_score: ["motorPowerScore", "motor_power_score", "autonomie_motor_power"],
  modes_assistance: ["modesAssistance", "mode_assistance"],
  tyres_efficiency_score: ["tyres_efficiency_score", "autonomie_pneus_score"],
  transmission_efficiency_score: [
    "transmission_efficiency_score",
    "autonomie_transmission_score",
  ],
  ergonomie: ["ergonomie_score"],
  roadPerformanceScore: ["road_performance_score", "roadPerformance"],
  offroadPerformanceScore: ["offroad_performance_score", "offroadPerformance"],
  utilityComfortScore: ["utility_comfort_score", "utilityComfort"],
};


async function fetchJSON(url, { optional = false, errorLabel = url } = {}) {
  let response;
  try {
    response = await fetch(url, { cache: "no-store" });
  } catch (err) {
    if (optional) return null;
    throw new Error(`Impossible de charger ${errorLabel}`);
  }

  if (!response.ok) {
    if (optional) return null;
    throw new Error(`Impossible de charger ${errorLabel} (${response.status})`);
  }

  try {
    return await response.json();
  } catch (err) {
    if (optional) return null;
    throw new Error(`Réponse JSON invalide pour ${errorLabel}`);
  }
}

async function init() {
  try {
    const [
      raw,
      metadata,
      componentTables,
      filtersConfig,
      useCaseWeights,
    ] = await Promise.all([
      fetchJSON("public/bikes.json", { errorLabel: "le dataset" }),
      fetchJSON("public/brand-info.json", {
        optional: true,
        errorLabel: "les métadonnées des marques",
      }),
      fetchJSON("scoring/component_quality.json", {
        optional: true,
        errorLabel: "les tables de qualité des composants",
      }),
      fetchJSON("scoring/filter_definitions.json", {
        errorLabel: "la configuration des filtres",
      }),
      fetchJSON("scoring/use_case_weights.json", {
        errorLabel: "la configuration des pondérations d'usages",
      }),
    ]);

    if (metadata === null) {
      console.warn("Métadonnées des marques indisponibles (public/brand-info.json)");
    } else if (Array.isArray(metadata)) {
      setBrandMetadata(metadata);
    } else if (metadata && !Array.isArray(metadata)) {
      console.warn("Format inattendu pour public/brand-info.json (tableau attendu)");
    }

    if (componentTables === null) {
      console.warn(
        "Tables de qualité des composants indisponibles (scoring/component_quality.json)"
      );
    } else if (componentTables && typeof componentTables === "object") {
      setComponentQualityTables(componentTables);
    } else if (componentTables !== undefined) {
      console.warn(
        "Format inattendu pour scoring/component_quality.json (objet attendu)"
      );
    }

    const weightsLoaded = setUseCaseWeightConfig(useCaseWeights);
    if (!weightsLoaded) {
      throw new Error(
        "Configuration des pondérations invalide (scoring/use_case_weights.json)"
      );
    }

    const filtersLoaded = setFilterDefinitions(filtersConfig);
    if (!filtersLoaded) {
      throw new Error("Configuration des filtres invalide (scoring/filter_definitions.json)");
    }

    const missingFilters = REQUIRED_FILTER_KEYS.filter(key => !hasFilterEntries(key));
    const optionalMissing = missingFilters.filter(key => OPTIONAL_FILTER_KEYS.has(key));
    const blockingMissing = missingFilters.filter(key => !OPTIONAL_FILTER_KEYS.has(key));

    if (optionalMissing.length) {
      console.warn(
        `Filtres optionnels ignorés : ${optionalMissing.join(", ")}`
      );
    }

    if (blockingMissing.length) {
      throw new Error(
        `Configuration des filtres incomplète (sections manquantes : ${blockingMissing.join(", ")})`
      );
    }

    const dataset = normalizeBikeDataset(raw);
    componentIdMap = dataset.componentIds;
    buildComponentKeyIndex(componentIdMap);
    initialData = dataset.bikes.map(normalizeBike);
    state.data = cloneData(initialData);
    const initialMetrics = state.selectedUseCases.size
      ? Array.from(getMetricsForUseCases(state.selectedUseCases))
      : state.selectedGlobalMetrics;
    recomputeGlobalScores(initialMetrics);
    updateUseCaseSummary(state.selectedGlobalMetrics);
    updateUseCaseUI();
    updateSortButtons();
    render();
  } catch (err) {
    console.error(err);
    resultsCount.textContent = "Erreur lors du chargement du dataset";
    tableBody.innerHTML = `<tr><td colspan="5" class="empty">${err.message}</td></tr>`;
  }
}

function defineSourceReference(target, sourceValue) {
  if (!target) return;
  if (sourceValue === undefined) return;
  Object.defineProperty(target, "__source", {
    value: sourceValue,
    enumerable: false,
    configurable: true,
    writable: true,
  });
}

function getSourceReference(row) {
  if (!row) return undefined;
  const descriptor = Object.getOwnPropertyDescriptor(row, "__source");
  return descriptor ? descriptor.value : undefined;
}

function normalizeBike(obj = {}) {
  const category = CATEGORY_SET.has(obj.category) ? obj.category : "urbain";
  const score = parseScore(obj.fiabiliteScore, obj.componentsScore);
  const id = obj.id ? String(obj.id) : generateId();
  const normalized = {
    id,
    brand: String(obj.brand ?? ""),
    model: String(obj.model ?? ""),
    category,
    url: obj.url ? String(obj.url) : null,
    fiabiliteScore: score,
  };

  const brandWebsite =
    getStringValue(obj, [
      "brandWebsite",
      "brand_website",
      "brandUrl",
      "brand_url",
      "companyUrl",
      "company_url",
      "manufacturerUrl",
      "manufacturer_url",
    ]) ?? null;
  const brandLegalName =
    getStringValue(obj, [
      "brandLegalName",
      "brand_legal_name",
      "companyName",
      "company_name",
      "legalName",
    ]) ?? null;
  const foundedValue = getNumericValue(obj, [
    "brandFoundedYear",
    "brand_founded_year",
    "brandFounded",
    "brand_founded",
    "companyFoundedYear",
    "company_founded_year",
    "companyFounded",
    "company_founded",
    "foundedYear",
    "anneeCreation",
  ]);
  const foundedYear = normalizeFoundedYear(foundedValue);

  if (brandWebsite) {
    normalized.brandWebsite = brandWebsite;
  }
  if (brandLegalName && brandLegalName.toLowerCase() !== normalized.brand.toLowerCase()) {
    normalized.brandLegalName = brandLegalName;
  }
  if (foundedYear != null) {
    normalized.brandFoundedYear = foundedYear;
  }

  if (typeof obj.componentsScore === "number" && Number.isFinite(obj.componentsScore)) {
    normalized.componentsScore = clamp(obj.componentsScore, 0, 10);
  }

  defineSourceReference(normalized, obj);

  Object.entries(NUMERIC_FIELDS).forEach(([key, aliases]) => {
    const value = getNumericValue(obj, [key, ...aliases]);
    if (value != null) {
      normalized[key] = value;
    }
  });

  const autonomyScore = computeAutonomyScore(normalized, obj);
  if (autonomyScore != null) {
    normalized.autonomieScore = autonomyScore.score;
    normalized.autonomieDetails = autonomyScore;
    normalized.autonomieBreakdown = autonomyScore.breakdown;
  } else {
    const fallback =
      getNumericValue(obj, ["autonomieScore", "autonomyScore", "equipementScore", "equipmentScore"]);
    if (fallback != null) {
      normalized.autonomieScore = clamp(fallback, 0, 10);
    }
  }

  const roadDetails = computeRoadPerformanceDetails(normalized, obj);
  if (roadDetails) {
    normalized.roadPerformanceScore = roadDetails.score;
    normalized.roadPerformanceBreakdown = roadDetails.breakdown;
    normalized.roadPerformanceDetails = roadDetails;
  }

  const offroadDetails = computeOffroadPerformanceDetails(normalized, obj);
  if (offroadDetails) {
    normalized.offroadPerformanceScore = offroadDetails.score;
    normalized.offroadPerformanceBreakdown = offroadDetails.breakdown;
    normalized.offroadPerformanceDetails = offroadDetails;
  }

  const utilityDetails = computeUtilityComfortDetails(normalized, obj);
  if (utilityDetails) {
    normalized.utilityComfortScore = utilityDetails.score;
    normalized.utilityComfortBreakdown = utilityDetails.breakdown;
    normalized.utilityComfortDetails = utilityDetails;
  }

  const privateDetails = computePrivatePerformanceDetails(normalized, obj);
  if (privateDetails) {
    normalized.privatePerformanceScore = privateDetails.score;
    normalized.privatePerformanceBreakdown = privateDetails.breakdown;
    normalized.privatePerformanceDetails = privateDetails;
  }

  const winterDetails = computeWinterPerformanceDetails(normalized, obj);
  if (winterDetails) {
    normalized.winterPerformanceScore = winterDetails.score;
    normalized.winterPerformanceBreakdown = winterDetails.breakdown;
    normalized.winterPerformanceDetails = winterDetails;
  }

  const reliabilityDetails = computeReliabilityScore(normalized, obj);
  if (reliabilityDetails) {
    normalized.fiabiliteScore = reliabilityDetails.score;
    normalized.fiabiliteDetails = reliabilityDetails;
    normalized.fiabiliteBreakdown = reliabilityDetails.breakdown;
  }

  applyGlobalScoreToRow(normalized, DEFAULT_GLOBAL_METRIC_KEYS);

  return applyBrandMetadata(normalized);
}

function parseScore(fiabilite, fallback) {
  if (typeof fiabilite === "number" && Number.isFinite(fiabilite)) {
    return clamp(fiabilite, 0, 10);
  }
  if (typeof fiabilite === "string") {
    const parsed = parseNumericString(fiabilite);
    if (parsed != null) {
      return clamp(parsed, 0, 10);
    }
  }
  if (typeof fallback === "number" && Number.isFinite(fallback)) {
    return clamp(fallback, 0, 10);
  }
  if (typeof fallback === "string") {
    const parsed = parseNumericString(fallback);
    if (parsed != null) {
      return clamp(parsed, 0, 10);
    }
  }
  return null;
}

function cloneData(arr) {
  if (!Array.isArray(arr)) return [];
  return arr.map(item => {
    const cloned = { ...item };
    const sourceRef = getSourceReference(item);
    if (sourceRef !== undefined) {
      defineSourceReference(cloned, sourceRef);
    }
    return cloned;
  });
}

function parseNumericString(value) {
  if (typeof value !== "string") return null;
  const trimmed = value.trim();
  if (!trimmed) return null;

  const sanitized = trimmed.replace(/[\u00a0]/g, " ");
  const segments = [];
  let current = "";

  for (let i = 0; i < sanitized.length; i += 1) {
    const char = sanitized[i];
    if (/[-+0-9.,]/.test(char)) {
      current += char;
      continue;
    }
    if (char === " ") {
      if (!current) continue;
      let j = i + 1;
      while (j < sanitized.length && sanitized[j] === " ") j += 1;
      const next = sanitized[j];
      if (next && /[0-9]/.test(next)) {
        current += " ";
      } else {
        segments.push(current);
        current = "";
      }
      continue;
    }
    if (current) {
      segments.push(current);
      current = "";
    }
  }

  if (current) {
    segments.push(current);
  }

  if (!segments.length) return null;

  const tokens = [];
  segments.forEach(segment => {
    const parts = segment.trim().split(/\s+/).filter(Boolean);
    if (!parts.length) return;
    if (parts.length === 1) {
      tokens.push(parts[0]);
      return;
    }
    const shouldCombine = parts.slice(1).every(part => {
      const integerPart = part.split(/[.,]/)[0];
      return /^\d{3}$/.test(integerPart);
    });
    if (shouldCombine) {
      tokens.push(parts.join(""));
    } else {
      tokens.push(...parts);
    }
  });

  if (!tokens.length) return null;

  let candidate = null;
  let candidateScore = -Infinity;

  tokens.forEach((token, index) => {
    let numericToken = token;
    if (/^\d+-\d+$/.test(numericToken)) {
      const [a, b] = numericToken.split("-").map(Number);
      numericToken = String(Math.max(a, b));
    }
    if (numericToken.includes(",") && numericToken.includes(".")) {
      const lastComma = numericToken.lastIndexOf(",");
      const lastDot = numericToken.lastIndexOf(".");
      if (lastDot > lastComma) {
        numericToken = numericToken.replace(/,/g, "");
      } else {
        numericToken = numericToken.replace(/\./g, "");
        numericToken = numericToken.replace(",", ".");
      }
    } else if (numericToken.includes(",")) {
      const parts = numericToken.split(",");
      if (parts.length === 2 && /^\d{3}$/.test(parts[1])) {
        numericToken = parts.join("");
      } else {
        numericToken = numericToken.replace(/,/g, ".");
      }
    } else if (numericToken.includes(".")) {
      const parts = numericToken.split(".");
      if (parts.length === 2 && /^\d{3}$/.test(parts[1])) {
        numericToken = parts.join("");
      }
    }

    const parsed = Number(numericToken);
    if (!Number.isFinite(parsed)) return;

    const weight = index + (/[.,]/.test(token) ? tokens.length : 0);
    if (weight >= candidateScore) {
      candidateScore = weight;
      candidate = parsed;
    }
  });

  return candidate;
}

function getNumericValue(source, keys) {
  for (const key of keys) {
    if (!key) continue;
    if (Object.prototype.hasOwnProperty.call(source, key)) {
      const value = source[key];
      if (typeof value === "number" && Number.isFinite(value)) {
        return value;
      }
      if (typeof value === "string") {
        const parsed = parseNumericString(value);
        if (parsed != null) {
          return parsed;
        }
      }
    }
  }
  return null;
}

function getNumericValueFromSources(primary, secondary, keys) {
  return getNumericValue(primary, keys) ?? getNumericValue(secondary, keys);
}

function normalizeFoundedYear(value) {
  if (value == null) return null;
  const numeric = typeof value === "number" ? value : Number(value);
  if (!Number.isFinite(numeric)) return null;
  const year = Math.round(numeric);
  const currentYear = new Date().getFullYear();
  if (year < 1800 || year > currentYear + 1) return null;
  return year;
}

function getStringValue(source, keys) {
  if (!source) return null;
  for (const key of keys) {
    if (!key) continue;
    if (Object.prototype.hasOwnProperty.call(source, key)) {
      const value = source[key];
      if (typeof value === "string") {
        const trimmed = value.trim();
        if (trimmed) {
          return trimmed;
        }
      }
    }
  }
  return null;
}

function getStringValueFromSources(primary, secondary, keys) {
  return getStringValue(primary, keys) ?? getStringValue(secondary, keys);
}

function getBooleanValue(source, keys) {
  if (!source) return null;
  for (const key of keys) {
    if (!key) continue;
    if (Object.prototype.hasOwnProperty.call(source, key)) {
      const value = source[key];
      if (typeof value === "boolean") {
        return value;
      }
      if (typeof value === "number" && (value === 0 || value === 1)) {
        return Boolean(value);
      }
      if (typeof value === "string") {
        const normalized = value.trim().toLowerCase();
        if (!normalized) continue;
        if (["true", "oui", "yes", "y", "present", "présent", "available"].includes(normalized)) {
          return true;
        }
        if (["false", "non", "no", "absent"].includes(normalized)) {
          return false;
        }
      }
    }
  }
  return null;
}

function getBooleanValueFromSources(primary, secondary, keys) {
  const value = getBooleanValue(primary, keys);
  if (value != null) return value;
  return getBooleanValue(secondary, keys);
}

function rescaleScore(value, options) {
  if (value == null || !Number.isFinite(value)) return null;
  const { currentMin = 0, currentMax = 1, targetMin = 0, targetMax = 1 } = options ?? {};
  if (currentMax === currentMin) return clamp(targetMin, targetMin, targetMax);
  const ratio = (value - currentMin) / (currentMax - currentMin);
  const scaled = targetMin + ratio * (targetMax - targetMin);
  return clamp(scaled, targetMin, targetMax);
}

function generateId() {
  if (typeof crypto !== "undefined" && crypto.randomUUID) {
    return crypto.randomUUID();
  }
  return `row-${Math.random().toString(36).slice(2, 10)}`;
}

function clamp(value, min, max) {
  return Math.min(max, Math.max(min, value));
}

function computeAutonomyScore(source, originalSource = {}) {
  const metrics = getFilterEntries("autonomy");
  if (!metrics?.length) return null;

  let total = 0;
  let weightSum = 0;
  let maxWeighted = 0;
  const breakdown = [];

  metrics.forEach(metric => {
    if (!metric) return;

    const weight = Number.isFinite(metric.weight) ? metric.weight : 1;
    if (weight <= 0) return;

    const configuredMax = Number.isFinite(metric.maxScore) ? metric.maxScore : 5;
    const maxScore = configuredMax > 0 ? configuredMax : 5;

    const componentValues =
      metric.componentId != null
        ? collectComponentValuesById(metric.componentId, originalSource, source)
        : null;
    const missingComponent = Array.isArray(componentValues) && componentValues.length === 0;

    const resolver = typeof metric.resolver === "function" ? metric.resolver : null;
    let raw = resolver ? resolver(source, originalSource) : null;
    if (missingComponent) {
      raw = null;
    }
    const hasFallback = Object.prototype.hasOwnProperty.call(metric, "fallback");
    const baseValue = raw == null ? (hasFallback ? metric.fallback : 0) : raw;
    const numericValue = Number(baseValue);
    let clampedValue = Number.isFinite(numericValue) ? clamp(numericValue, 0, maxScore) : 0;
    if (missingComponent) {
      clampedValue = 0;
    }
    const weighted = clampedValue * weight;

    total += weighted;
    weightSum += weight;
    maxWeighted += maxScore * weight;
    const breakdownEntry = {
      key: metric.key,
      label: metric.label ?? metric.key,
      weight: Number(weight.toFixed(2)),
      score: Number(clampedValue.toFixed(2)),
      weighted: Number(weighted.toFixed(2)),
    };

    if (missingComponent) {
      breakdownEntry.rawValue = "-non défini";
    } else if (Array.isArray(componentValues) && componentValues.length) {
      breakdownEntry.rawValue = componentValues;
    }

    breakdown.push(breakdownEntry);
  });

  if (!weightSum || !maxWeighted) return null;
  const normalized = (total / maxWeighted) * 10;
  const normalizedScore = Number(normalized.toFixed(2));

  const detailedBreakdown = breakdown.map(item => ({
    ...item,
    contribution: Number(((item.weighted / maxWeighted) * 10).toFixed(2)),
  }));

  return {
    score: normalizedScore,
    weightSum: Number(weightSum.toFixed(2)),
    totalWeighted: Number(total.toFixed(2)),
    maxRawScore: Number(maxWeighted.toFixed(2)),
    breakdown: detailedBreakdown,
  };
}

function resolveBatteryCapacity(source, originalSource) {
  const directScore = getNumericValueFromSources(source, originalSource, [
    "battery_capacity_score",
    "batteryCapacityScore",
    "autonomie_batterie_capacite",
    "autonomie_batterie_score",
  ]);
  if (directScore != null && directScore <= 5) {
    return clamp(directScore, 0, 5);
  }

  const wattHours = getNumericValueFromSources(source, originalSource, [
    "batterie_cap_wh",
    "batteryWh",
    "battery_wh",
    "batteryCapacityWh",
  ]);
  if (wattHours != null) {
    if (wattHours < 500) return 1;
    if (wattHours < 700) return 2;
    if (wattHours < 900) return 3;
    if (wattHours < 1400) return 4;
    return 5;
  }

  const ampHours = getNumericValueFromSources(source, originalSource, ["batterie_cap_ah"]);
  if (ampHours != null) {
    return rescaleScore(ampHours, { currentMin: 1, currentMax: 3, targetMin: 1, targetMax: 5 });
  }

  return null;
}

function resolveBatteryCells(source, originalSource) {
  const directScore = getNumericValueFromSources(source, originalSource, [
    "battery_cells_score",
    "batteryCellsScore",
    "autonomie_batterie_cellules",
  ]);
  if (directScore != null && directScore <= 5) {
    return clamp(directScore, 0, 5);
  }

  const ul = getBooleanValueFromSources(originalSource, source, [
    "battery_certified_ul",
    "batterie_certif_ul",
  ]);
  if (ul === true) {
    return 5;
  }
  if (ul === false) {
    return 0;
  }

  const label = getStringValueFromSources(source, originalSource, [
    "battery_cells",
    "batteryCells",
    "cellules",
    "cellules_label",
  ]);
  if (label) {
    const value = label.toLowerCase();
    if (/(panasonic|ncr|sony|vtc)/.test(value)) return 5;
    if (/(samsung\s?50e|lg\s?mj1|mj1|50e)/.test(value)) return 4;
    if (/(lg|samsung)/.test(value)) return 3;
    if (/(no[-\s]?name|generic|inconnue|unknown)/.test(value)) return 1;
    return 2;
  }

  const numeric = getNumericValueFromSources(source, originalSource, ["cellules_type"]);
  if (numeric != null) {
    if (numeric <= 5 && numeric >= 0) {
      return rescaleScore(numeric, { currentMin: 0, currentMax: 3, targetMin: 1, targetMax: 5 });
    }
  }

  return null;
}

function resolveRangeExtender(source, originalSource) {
  const directScore = getNumericValueFromSources(source, originalSource, [
    "range_extender_score",
    "rangeExtenderScore",
    "autonomie_add_on",
    "battery_add_on",
    "bonus_add_on",
    "bonus_addon",
    "bonus_add-on",
    "bonusAddon",
  ]);
  if (directScore != null) {
    if (directScore <= 5) {
      return clamp(directScore, 0, 5);
    }
    return rescaleScore(directScore, { currentMin: 0, currentMax: 3, targetMin: 0, targetMax: 5 });
  }

  const text = getStringValueFromSources(source, originalSource, [
    "range_extender",
    "rangeExtender",
    "battery_addon",
  ]);
  if (text) {
    const normalized = text.toLowerCase();
    if (/(double|dual|intégré|intégrée|integrated|two batteries)/.test(normalized)) {
      return 5;
    }
    if (/(option|optionnel|available|addon)/.test(normalized)) {
      return 3;
    }
    if (/(aucun|none|sans)/.test(normalized)) {
      return 0;
    }
  }

  const flag = getBooleanValueFromSources(source, originalSource, [
    "hasRangeExtender",
    "range_extender",
    "rangeExtender",
    "battery_addon",
  ]);
  if (flag != null) {
    return flag ? 3 : 0;
  }

  return null;
}

function resolveMotorType(source, originalSource) {
  const directScore = getNumericValueFromSources(source, originalSource, [
    "motor_type_score",
    "motorTypeScore",
    "autonomie_motor_type",
  ]);
  if (directScore != null && directScore <= 5) {
    return clamp(directScore, 0, 5);
  }

  const label = getStringValueFromSources(source, originalSource, [
    "motor_type",
    "moteur_type_label",
  ]);
  if (label) {
    const value = label.toLowerCase();
    if (/(front)/.test(value)) return 1;
    if (/(direct drive)/.test(value)) return 2;
    if (/(rear|hub)/.test(value)) return 3;
    if (/(mid)/.test(value)) {
      return /(bosch|yamaha|shimano|brose)/.test(value) ? 5 : 4;
    }
  }

  const numeric = getNumericValueFromSources(source, originalSource, ["moteur_type"]);
  if (numeric != null) {
    if (numeric <= 5 && numeric >= 0) {
      if (numeric === 1) return 1;
      if (numeric === 2) return 5;
      if (numeric === 3) return 3;
      return rescaleScore(numeric, { currentMin: 1, currentMax: 3, targetMin: 1, targetMax: 5 });
    }
  }

  return null;
}

function resolveMotorPower(source, originalSource) {
  const directScore = getNumericValueFromSources(source, originalSource, [
    "motor_power_score",
    "motorPowerScore",
    "autonomie_motor_power",
  ]);
  if (directScore != null && directScore <= 5) {
    return clamp(directScore, 0, 5);
  }

  const watts = getNumericValueFromSources(source, originalSource, [
    "motor_power_w",
    "motorPowerW",
    "puissance_w",
    "motor_nominal_power",
  ]);
  if (watts != null) {
    const absolute = Math.abs(watts);
    if (absolute >= 1000) return 1;
    if (absolute >= 750) return 2;
    if (absolute >= 500) return 3;
    if (absolute >= 350) return 4;
    return 5;
  }

  const numeric = getNumericValueFromSources(source, originalSource, ["puissance"]);
  if (numeric != null) {
    if (numeric <= 5 && numeric >= 0) {
      if (numeric === 1) return 5;
      if (numeric === 2) return 3;
      if (numeric === 3) return 1;
      return rescaleScore(numeric, { currentMin: 0, currentMax: 3, targetMin: 1, targetMax: 5 });
    }
  }

  return null;
}

function resolveFrameEfficiency(source, originalSource) {
  const directScore = getNumericValueFromSources(source, originalSource, [
    "frame_efficiency_score",
    "frameEfficiencyScore",
    "autonomie_cadre_score",
  ]);
  if (directScore != null && directScore <= 5) {
    return clamp(directScore, 0, 5);
  }

  const label = getStringValueFromSources(source, originalSource, [
    "cadre_type",
    "frame_type",
  ]);
  if (label) {
    const value = label.toLowerCase();
    if (/(fatbike)/.test(value)) return 1;
    if (/(enduro|full\s?suspension|fs)/.test(value)) return 2;
    if (/(trail|hardtail)/.test(value)) return 3;
    if (/(semi-rigide|semi rigide|semi)/.test(value)) return 4;
    if (/(rigide|gravel|urbain|commuter)/.test(value)) return 5;
  }

  const numeric = getNumericValueFromSources(source, originalSource, ["cadre_susp"]);
  if (numeric != null) {
    if (numeric <= 5 && numeric >= 0) {
      if (numeric === 1) return 5;
      if (numeric === 2) return 3;
      if (numeric === 3) return 1;
      return rescaleScore(numeric, { currentMin: 1, currentMax: 3, targetMin: 1, targetMax: 5 });
    }
  }

  return null;
}

function resolveTyres(source, originalSource) {
  const directScore = getNumericValueFromSources(source, originalSource, [
    "tyres_efficiency_score",
    "autonomie_pneus_score",
  ]);
  if (directScore != null && directScore <= 5) {
    return clamp(directScore, 0, 5);
  }

  const values = collectValuesFromSources(originalSource, source, TYRES_FIELDS);
  const widthInches = extractTyreWidthInches(values);
  if (widthInches != null) {
    const score = mapTyreWidthToScore(widthInches);
    if (score != null) return score;
  }

  for (const value of values) {
    if (typeof value === "string") {
      const lower = value.toLowerCase();
      if (/(fatbike|4\"|5\"|26x4|27\.5x3|4\.0x|4\.5x|5\.0x)/.test(lower)) return 1;
      if (/(vtt|mtb|knob|crampon|trail)/.test(lower)) return 2;
      if (/(hybrid|cst|vee|kenda|all[-\s]?terrain|mixed)/.test(lower)) return 3;
      if (/(slick|route|road|marathon)/.test(lower)) return 4;
      if (/(contact|continental|schwalbe|big ben|marathon plus)/.test(lower)) return 5;
    }
    if (typeof value === "number" && Number.isFinite(value)) {
      if (value >= 0 && value <= 5) {
        return clamp(value, 0, 5);
      }
    }
  }

  return null;
}

function mapTransmissionSpeedsToScore(speeds) {
  if (speeds == null || Number.isNaN(speeds)) return null;
  const value = Number(speeds);
  if (!Number.isFinite(value)) return null;
  if (value <= 0) return 0;
  const clampedValue = clamp(value, 3, 12);
  const scaled = rescaleScore(clampedValue, {
    currentMin: 3,
    currentMax: 12,
    targetMin: 1,
    targetMax: 5,
  });
  if (scaled == null) return null;
  return Number(scaled.toFixed(2));
}

function resolveTransmission(source, originalSource) {
  const directScore = getNumericValueFromSources(source, originalSource, [
    "transmission_efficiency_score",
    "autonomie_transmission_score",
  ]);
  if (directScore != null && directScore <= 5) {
    return clamp(directScore, 0, 5);
  }

  const speedsValue = getNumericValueFromSources(source, originalSource, [
    "derailleur_speeds",
    "vitesses_arriere",
    "vitesses",
    "rear_speeds",
    "speeds",
  ]);
  if (speedsValue != null) {
    const score = mapTransmissionSpeedsToScore(speedsValue);
    if (score != null) return score;
  }

  const values = collectValuesFromSources(source, originalSource, TRANSMISSION_FIELDS);

  for (const value of values) {
    if (typeof value === "number" && Number.isFinite(value)) {
      const score = mapTransmissionSpeedsToScore(value);
      if (score != null) return score;
    }
    if (typeof value === "string") {
      const match = value.match(/(\d{1,2})\s*(?:vitesses?|vit|spd|speeds?)/i);
      if (match) {
        const parsed = Number(match[1]);
        const score = mapTransmissionSpeedsToScore(parsed);
        if (score != null) return score;
      }
      const genericMatch = value.match(/\b(\d{1,2})\b/);
      if (genericMatch) {
        const parsed = Number(genericMatch[1]);
        if (parsed >= 3 && parsed <= 20) {
          const score = mapTransmissionSpeedsToScore(parsed);
          if (score != null) return score;
        }
      }
    }
  }

  for (const value of values) {
    if (typeof value !== "string") continue;
    const lower = value.toLowerCase();
    if (/(tourney|altus)/.test(lower)) return 2;
    if (/(alivio|deore|nx)/.test(lower)) return 3;
    if (/(slx|xt|gx)/.test(lower)) return 4;
    if (/(xt\s?12|eagle|x01|xx1|rohloff|enviolo)/.test(lower)) return 5;
  }

  return null;
}

function computeRoadPerformanceDetails(source, originalSource = {}) {
  return computePerformanceDetails(source, originalSource, getFilterEntries("road_performance"));
}

function computeOffroadPerformanceDetails(source, originalSource = {}) {
  return computePerformanceDetails(source, originalSource, getFilterEntries("offroad_performance"));
}

function computeUtilityComfortDetails(source, originalSource = {}) {
  return computePerformanceDetails(source, originalSource, getFilterEntries("utility_comfort"));
}

function computePrivatePerformanceDetails(source, originalSource = {}) {
  return computePerformanceDetails(source, originalSource, getFilterEntries("private_performance"));
}

function computeWinterPerformanceDetails(source, originalSource = {}) {
  return computePerformanceDetails(source, originalSource, getFilterEntries("winter_performance"));
}

function computePerformanceDetails(source, originalSource, subCategories) {
  if (!Array.isArray(subCategories) || !subCategories.length) return null;

  const breakdown = [];
  let totalWeighted = 0;
  let maxWeighted = 0;
  let weightSum = 0;

  subCategories.forEach(sub => {
    if (!sub) return;

    const weight = Number.isFinite(sub.weight) ? sub.weight : 1;
    if (weight <= 0) return;

    const configuredMax = Number.isFinite(sub.maxScore) ? sub.maxScore : 5;
    const maxScore = configuredMax > 0 ? configuredMax : 5;

    const componentValues =
      sub.componentId != null
        ? collectComponentValuesById(sub.componentId, originalSource, source)
        : null;
    const missingComponent = Array.isArray(componentValues) && componentValues.length === 0;

    const resolver = typeof sub.resolver === "function" ? sub.resolver : null;
    let raw = resolver ? resolver(source, originalSource) : null;
    if (missingComponent) {
      raw = null;
    }
    const hasFallback = Object.prototype.hasOwnProperty.call(sub, "fallback");
    const baseValue = raw == null ? (hasFallback ? sub.fallback : 0) : raw;
    const numericValue = Number(baseValue);
    let clampedValue = Number.isFinite(numericValue) ? clamp(numericValue, 0, maxScore) : 0;
    if (missingComponent) {
      clampedValue = 0;
    }
    const weighted = clampedValue * weight;

    const entry = {
      key: sub.key,
      label: sub.label ?? sub.key,
      score: Number(clampedValue.toFixed(2)),
      weight: Number(weight.toFixed(2)),
      weighted: Number(weighted.toFixed(2)),
      maxScore: Number(maxScore.toFixed(2)),
    };

    if (missingComponent) {
      entry.rawValue = "-non défini";
    } else if (Array.isArray(componentValues) && componentValues.length) {
      entry.rawValue = componentValues;
    }

    breakdown.push(entry);
    totalWeighted += weighted;
    maxWeighted += maxScore * weight;
    weightSum += weight;
  });

  if (!maxWeighted) return null;

  const normalized = clamp(1 + (totalWeighted / maxWeighted) * 9, 1, 10);
  const detailedBreakdown = breakdown.map(item => ({
    ...item,
    contribution: Number(((item.weighted / maxWeighted) * 10).toFixed(2)),
  }));

  return {
    score: Number(normalized.toFixed(2)),
    rawScore: Number(totalWeighted.toFixed(2)),
    maxRawScore: Number(maxWeighted.toFixed(2)),
    totalWeighted: Number(totalWeighted.toFixed(2)),
    weightSum: Number(weightSum.toFixed(2)),
    breakdown: detailedBreakdown,
  };
}

function getFirstValueFromSources(primary, secondary, keys) {
  if (!Array.isArray(keys)) return null;
  for (const key of keys) {
    if (primary && Object.prototype.hasOwnProperty.call(primary, key)) {
      const value = primary[key];
      if (value !== undefined && value !== null) {
        return value;
      }
    }
    if (secondary && Object.prototype.hasOwnProperty.call(secondary, key)) {
      const value = secondary[key];
      if (value !== undefined && value !== null) {
        return value;
      }
    }
  }
  return null;
}

function collectValuesFromSources(primary, secondary, keys) {
  if (!Array.isArray(keys) || !keys.length) return [];
  const values = [];
  const seen = new Set();
  const addValue = value => {
    if (value === undefined || value === null) return;
    if (typeof value === "string") {
      const trimmed = value.trim();
      if (!trimmed) return;
      const marker = trimmed.toLowerCase();
      if (seen.has(marker)) return;
      seen.add(marker);
      values.push(trimmed);
      return;
    }
    if (typeof value === "number" && Number.isFinite(value)) {
      const marker = `num:${value}`;
      if (seen.has(marker)) return;
      seen.add(marker);
      values.push(value);
      return;
    }
    if (typeof value === "boolean") {
      const marker = `bool:${value}`;
      if (seen.has(marker)) return;
      seen.add(marker);
      values.push(value);
      return;
    }
    if (Array.isArray(value)) {
      value.forEach(addValue);
      return;
    }
    if (value && typeof value === "object") {
      Object.values(value).forEach(addValue);
    }
  };

  [primary, secondary].forEach(source => {
    if (!source) return;
    keys.forEach(key => {
      if (!key) return;
      if (Object.prototype.hasOwnProperty.call(source, key)) {
        addValue(source[key]);
      }
    });
  });

  return values;
}

function normalizeLabel(value) {
  if (value == null) return "";
  const text = String(value)
    .normalize("NFKD")
    .replace(/[̀-ͯ]/g, "")
    .toLowerCase()
    .replace(/[^a-z0-9]+/g, " ")
    .trim();
  return text;
}

const BOOLEAN_TRUE_LABELS = new Set(["oui", "yes", "true", "vrai", "present", "avec", "inclus", "presente"]);
const BOOLEAN_FALSE_LABELS = new Set(["non", "no", "false", "faux", "sans", "absent", "absente", "aucun"]);

function interpretBooleanLike(value) {
  if (typeof value === "boolean") return value;
  if (typeof value === "number") {
    if (value === 1) return true;
    if (value === 0) return false;
  }
  if (typeof value === "string") {
    const label = normalizeLabel(value);
    if (!label) return null;
    if (BOOLEAN_TRUE_LABELS.has(label)) return true;
    if (BOOLEAN_FALSE_LABELS.has(label)) return false;
    const tokens = label.split(/\s+/).filter(Boolean);
    if (tokens.some(token => BOOLEAN_TRUE_LABELS.has(token))) return true;
    if (tokens.some(token => BOOLEAN_FALSE_LABELS.has(token))) return false;
  }
  return null;
}

function extractNumeric(value) {
  if (value == null) return null;
  if (typeof value === "number") {
    return Number.isFinite(value) ? value : null;
  }
  const text = String(value).trim().toLowerCase().replace(/,/g, ".");
  const match = text.match(/[-+]?\d*\.?\d+/);
  if (!match) return null;
  const parsed = Number(match[0]);
  return Number.isFinite(parsed) ? parsed : null;
}

function coerceDirectRating(value, maxScore) {
  if (value == null) return null;
  if (typeof value === "boolean") return null;
  if (typeof value === "number") {
    const numeric = Number(value);
    if (Number.isFinite(numeric) && numeric >= 0 && numeric <= maxScore) {
      return numeric;
    }
    return null;
  }
  if (typeof value === "string") {
    const numeric = extractNumeric(value);
    if (numeric != null && numeric >= 0 && numeric <= maxScore) {
      return numeric;
    }
  }
  return null;
}

function lookupComponentScore(key, raw, maxScore = 5) {
  if (!componentQualityTables || !key) return null;
  const tableEntry = componentQualityTables[key];
  if (!tableEntry) return null;

  const limit = Number.isFinite(maxScore) && maxScore > 0
    ? maxScore
    : Number.isFinite(tableEntry.maxScore) && tableEntry.maxScore > 0
      ? tableEntry.maxScore
      : 10;

  const values = tableEntry.values ?? {};
  const defaultScore = tableEntry.defaultScore;
  const seen = new Set();

  const tryLabel = candidate => {
    if (candidate === undefined || candidate === null) {
      return null;
    }

    let normalized;
    if (typeof candidate === "boolean") {
      normalized = candidate ? "true" : "false";
    } else {
      normalized = normalizeLabel(String(candidate));
    }

    if (!normalized || seen.has(normalized)) {
      return null;
    }

    seen.add(normalized);
    const lookup = values[normalized];
    if (lookup == null) {
      return null;
    }

    const numeric = Number(lookup);
    if (!Number.isFinite(numeric)) {
      return null;
    }
    return clamp(numeric, 0, limit);
  };

  const tryValue = value => {
    if (value === undefined || value === null) {
      return null;
    }

    if (Array.isArray(value)) {
      for (const item of value) {
        const score = tryValue(item);
        if (score != null) {
          return score;
        }
      }
      return null;
    }

    if (value && typeof value === "object") {
      for (const item of Object.values(value)) {
        const score = tryValue(item);
        if (score != null) {
          return score;
        }
      }
      return null;
    }

    let score = tryLabel(value);
    if (score != null) {
      return score;
    }

    const text = String(value).trim();
    if (text) {
      const parts = text
        .split(/[\/,;|+&]/)
        .map(part => part.trim())
        .filter(Boolean);
      for (const part of parts) {
        score = tryLabel(part);
        if (score != null) {
          return score;
        }
      }

      const tokens = text.split(/\s+/).filter(Boolean);
      if (tokens.length > 1) {
        score = tryLabel(tokens[0]);
        if (score != null) {
          return score;
        }
        score = tryLabel(tokens.slice(0, 2).join(" "));
        if (score != null) {
          return score;
        }
      }
    }

    const numericMatches = String(value).match(/\d+(?:\.\d+)?/g);
    if (numericMatches) {
      for (const match of numericMatches) {
        score = tryLabel(match);
        if (score != null) {
          return score;
        }
      }
    }

    return null;
  };

  const resolved = tryValue(raw);
  if (resolved != null) {
    return resolved;
  }

  if (defaultScore != null) {
    const numeric = Number(defaultScore);
    if (Number.isFinite(numeric)) {
      return clamp(numeric, 0, limit);
    }
  }

  return null;
}

function getComponentTableMaxScore(key, fallback = 5) {
  if (!componentQualityTables || !key) return fallback;
  const tableEntry = componentQualityTables[key];
  if (!tableEntry) return fallback;
  const numeric = Number(tableEntry.maxScore);
  if (!Number.isFinite(numeric) || numeric <= 0) {
    return fallback;
  }
  return numeric;
}

function getComponentTableDefaultScore(key) {
  if (!componentQualityTables || !key) return null;
  const tableEntry = componentQualityTables[key];
  if (!tableEntry) return null;
  const numeric = Number(tableEntry.defaultScore);
  if (!Number.isFinite(numeric)) {
    return null;
  }
  return numeric;
}

function getComponentScoreFromLabels(category, raw, maxScore = 5) {
  if (!category || raw == null) return null;
  return lookupComponentScore(category, raw, maxScore);
}

function mapBatteryCapacityToLabel(wh) {
  if (wh == null || Number.isNaN(wh)) return null;
  const value = Number(wh);
  if (!Number.isFinite(value) || value <= 0) return null;
  if (value >= 1300) return "1300–2990 Wh";
  if (value >= 1000) return "1000–1299 Wh";
  if (value >= 800) return "800–999 Wh";
  if (value >= 500) return "500–799 Wh";
  return "<500 Wh";
}

function mapWeightToLabel(weightKg) {
  if (weightKg == null || Number.isNaN(weightKg)) return null;
  const value = Number(weightKg);
  if (!Number.isFinite(value) || value <= 0) return null;
  if (value <= 20) return "≤ 20 kg";
  if (value <= 23) return "21–23 kg";
  if (value <= 26) return "24–26 kg";
  if (value <= 29) return "27–29 kg";
  return "≥ 30 kg";
}

function mapTyreWidthToScore(widthInches) {
  if (widthInches == null || Number.isNaN(widthInches)) return null;
  const value = Number(widthInches);
  if (!Number.isFinite(value) || value <= 0) return null;
  const clampedWidth = clamp(value, 1, 4);
  const ratio = (clampedWidth - 1) / 3;
  const score = 5 - ratio * 4;
  return Number(score.toFixed(2));
}

function extractTyreWidthInches(values) {
  if (!Array.isArray(values) || !values.length) return null;
  const widths = [];

  values.forEach(value => {
    if (typeof value === "number" && Number.isFinite(value)) {
      if (value > 0 && value < 10) {
        widths.push(value);
        return;
      }
      if (value >= 10 && value <= 150) {
        const converted = value / 25.4;
        if (converted >= 0.8 && converted <= 6.5) {
          widths.push(converted);
        }
      }
      return;
    }

    if (typeof value !== "string") return;
    const sanitized = value.replace(/[,;]/g, " ").replace(/\s+/g, " ").trim();
    if (!sanitized) return;

    const matches = sanitized.match(/\d+(?:[.,]\d+)?/g);
    if (!matches) return;

    matches.forEach(match => {
      const numeric = Number(match.replace(/,/g, "."));
      if (!Number.isFinite(numeric) || numeric <= 0) return;

      let width = numeric;
      if (width > 10) {
        if (width > 150) return;
        width = width / 25.4;
      }

      if (width >= 0.8 && width <= 6.5) {
        widths.push(width);
      }
    });
  });

  if (!widths.length) return null;
  return Math.max(...widths);
}

function resolveReliabilityMotor(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Moteur");
  const values = collectValuesFromSources(originalSource, source, [
    "motor_brand",
    "motorBrand",
    "motor",
    "motor_label",
    "motor_model",
    "moteur_marque",
    "motor_name",
  ]);
  return getComponentScoreFromLabels("Moteur", values, maxScore);
}

function resolveReliabilityController(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Controleur");
  const values = collectValuesFromSources(originalSource, source, [
    "controller",
    "controleur",
    "controller_label",
  ]);
  return getComponentScoreFromLabels("Controleur", values, maxScore);
}

function resolveReliabilityDisplay(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Display");
  const values = collectValuesFromSources(originalSource, source, [
    "display",
    "afficheur",
    "screen",
  ]);
  return getComponentScoreFromLabels("Display", values, maxScore);
}

function resolveReliabilityDerailleur(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Derailleur");
  const values = collectValuesFromSources(originalSource, source, [
    "derailleur_label",
    "derailleur",
    "transmission",
  ]);
  return getComponentScoreFromLabels("Derailleur", values, maxScore);
}

function resolveReliabilityBrakes(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Freins");
  const values = collectValuesFromSources(originalSource, source, [
    "brakes_label",
    "freins",
    "brakes",
  ]);
  return getComponentScoreFromLabels("Freins", values, maxScore);
}

function resolveReliabilityTyres(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Pneus");
  const values = collectValuesFromSources(originalSource, source, [
    "tyres_label",
    "pneus",
    "tires",
  ]);
  return getComponentScoreFromLabels("Pneus", values, maxScore);
}

function resolveReliabilityBatteryCapacity(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Batterie_capacite");
  const capacity = getNumericValueFromSources(source, originalSource, [
    "batteryWh",
    "battery_wh",
    "batteryCapacityWh",
    "battery_capacity_wh",
    "battery_capacity",
    "batterie_cap_wh",
  ]);
  const label = mapBatteryCapacityToLabel(capacity);
  if (!label) return null;
  return getComponentScoreFromLabels("Batterie_capacite", label, maxScore);
}

function resolveReliabilityBatteryCertification(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Batterie_certification");
  const ul = getBooleanValueFromSources(originalSource, source, [
    "battery_certified_ul",
    "batterie_certif_ul",
  ]);
  if (ul === true) {
    return maxScore > 0 ? maxScore : 5;
  }
  if (ul === false) {
    return 0;
  }

  const label = getStringValueFromSources(originalSource, source, [
    "battery_certification",
    "batterie_certification",
    "battery_certif",
  ]);
  if (label) {
    const score = getComponentScoreFromLabels("Batterie_certification", label, maxScore);
    if (score != null) return score;
  }
  return null;
}

function isSuspensionValueAbsent(value) {
  if (value == null) return false;
  if (typeof value === "boolean") return value === false;
  if (typeof value === "number") return value === 0;
  if (typeof value === "string") {
    const label = normalizeLabel(value);
    if (!label) return false;
    return (
      label === "aucune" ||
      label === "sans" ||
      label === "rigide" ||
      label.includes("sans suspension") ||
      label.includes("no suspension") ||
      label.includes("none") ||
      label.includes("rigide")
    );
  }
  if (Array.isArray(value)) {
    return value.some(isSuspensionValueAbsent);
  }
  if (typeof value === "object") {
    return Object.values(value).some(isSuspensionValueAbsent);
  }
  return false;
}

function resolveReliabilityBatteryCells(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Cellules_batterie");
  const values = collectValuesFromSources(originalSource, source, [
    "battery_cells",
    "batteryCells",
    "cellules_type",
    "cellules",
  ]);
  if (!values.length) return null;
  const score = getComponentScoreFromLabels("Cellules_batterie", values, maxScore);
  if (score != null) return score;
  const first = values.find(value => typeof value === "string");
  if (first) {
    const brand = first.split(/\s+/)[0];
    if (brand) {
      return getComponentScoreFromLabels("Cellules_batterie", brand, maxScore);
    }
  }
  return null;
}

function resolveReliabilityWeight(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Poids_avec_batterie");
  const weight = getNumericValueFromSources(source, originalSource, [
    "weight",
    "weight_kg",
    "poids",
  ]);
  const label = mapWeightToLabel(weight);
  if (!label) return null;
  return getComponentScoreFromLabels("Poids_avec_batterie", label, maxScore);
}

function resolveReliabilityStem(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Potence_ajustable", 1);
  const value = getBooleanValueFromSources(originalSource, source, [
    "stem_adjustable",
    "potence_ajustable",
  ]);
  if (value == null) return null;
  return getComponentScoreFromLabels("Potence_ajustable", value ? "True" : "False", maxScore);
}

function resolveReliabilityFrontSuspension(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Suspension_avant");
  const values = collectValuesFromSources(originalSource, source, FRONT_SUSPENSION_FIELDS);
  if (!values.length) return null;

  if (values.some(isSuspensionValueAbsent)) {
    return maxScore > 0 ? maxScore : 5;
  }

  return getComponentScoreFromLabels("Suspension_avant", values, maxScore);
}

function resolveReliabilityRearSuspension(source, originalSource) {
  const maxScore = getComponentTableMaxScore("Suspension_arriere");
  const values = collectValuesFromSources(originalSource, source, REAR_SUSPENSION_FIELDS);
  if (!values.length) return null;

  if (values.some(isSuspensionValueAbsent)) {
    return maxScore > 0 ? maxScore : 5;
  }

  return getComponentScoreFromLabels("Suspension_arriere", values, maxScore);
}

function computeReliabilityScore(source, originalSource = {}) {
  if (!componentQualityTables) return null;
  let totalWeighted = 0;
  let maxWeighted = 0;
  const breakdown = [];

  const metrics = getFilterEntries("reliability");
  if (!metrics?.length) return null;

  metrics.forEach(metric => {
    if (!metric) return;

    const weight = Number.isFinite(metric.weight) ? metric.weight : 1;
    if (weight <= 0) return;

    const configuredMax = Number.isFinite(metric.maxScore)
      ? metric.maxScore
      : getComponentTableMaxScore(metric.qualityKey ?? metric.category, 5);
    const inferredMax = configuredMax > 0 ? configuredMax : 5;

    const componentValues =
      metric.componentId != null
        ? collectComponentValuesById(metric.componentId, originalSource, source)
        : null;
    const missingComponent = Array.isArray(componentValues) && componentValues.length === 0;

    const resolver = typeof metric.resolver === "function" ? metric.resolver : null;
    let raw = resolver ? resolver(source, originalSource) : null;
    if (missingComponent) {
      raw = null;
    }
    const hasFallback = Object.prototype.hasOwnProperty.call(metric, "fallback");
    const baseValue = raw == null ? (hasFallback ? metric.fallback : 0) : raw;
    const numericValue = Number(baseValue);
    let clamped = Number.isFinite(numericValue) ? clamp(numericValue, 0, inferredMax) : 0;
    if (missingComponent) {
      clamped = 0;
    }
    const weighted = clamped * weight;

    totalWeighted += weighted;
    maxWeighted += inferredMax * weight;
    const entry = {
      key: metric.key,
      label: metric.label ?? metric.key,
      score: Number(clamped.toFixed(2)),
      weight: Number(weight.toFixed(2)),
      weighted: Number(weighted.toFixed(2)),
    };

    if (missingComponent) {
      entry.rawValue = "-non défini";
    } else if (Array.isArray(componentValues) && componentValues.length) {
      entry.rawValue = componentValues;
    }

    breakdown.push(entry);
  });

  if (!maxWeighted) return null;

  const normalized = clamp(1 + (totalWeighted / maxWeighted) * 9, 1, 10);
  const detailedBreakdown = breakdown.map(item => ({
    ...item,
    contribution: Number(((item.weighted / maxWeighted) * 10).toFixed(2)),
  }));
  const equivalentWeightSum = maxWeighted / 5;

  return {
    score: Number(normalized.toFixed(2)),
    totalWeighted: Number(totalWeighted.toFixed(2)),
    weightSum: Number(equivalentWeightSum.toFixed(2)),
    maxRawScore: Number(maxWeighted.toFixed(2)),
    breakdown: detailedBreakdown,
  };
}

function getMetricLabel(key) {
  return GLOBAL_METRIC_DEFINITIONS[key]?.label ?? key;
}

function getMetricColor(key, index = 0) {
  if (METRIC_COLOR_MAP[key]) {
    return METRIC_COLOR_MAP[key];
  }
  const fallback = METRIC_COLOR_FALLBACKS[index % METRIC_COLOR_FALLBACKS.length];
  return fallback ?? "#0ea5e9";
}

function normalizeMetricKeys(metricKeys) {
  const keys = Array.isArray(metricKeys) ? metricKeys : DEFAULT_GLOBAL_METRIC_KEYS;
  const filtered = keys.filter(key => GLOBAL_METRIC_DEFINITIONS[key]);
  if (!filtered.length) {
    return [...DEFAULT_GLOBAL_METRIC_KEYS];
  }
  return Array.from(new Set(filtered));
}

function computeGlobalScore(row = {}, metricKeys = DEFAULT_GLOBAL_METRIC_KEYS) {
  const keys = normalizeMetricKeys(metricKeys);
  const entries = keys
    .map(key => {
      const rawValue = row?.[key];
      if (typeof rawValue !== "number" || !Number.isFinite(rawValue)) {
        return null;
      }
      const value = roundScoreValue(rawValue, 1);
      if (value == null) {
        return null;
      }
      const weight = getMetricWeight(key);
      return {
        key,
        label: getMetricLabel(key),
        value,
        weight,
      };
    })
    .filter(Boolean);

  if (!entries.length) return null;

  const totalWeight = entries.reduce((acc, item) => acc + item.weight, 0);
  if (totalWeight <= 0) return null;

  const weightedSum = entries.reduce(
    (acc, item) => acc + item.value * item.weight,
    0
  );
  const average = weightedSum / totalWeight;

  return {
    score: Number(average.toFixed(2)),
    breakdown: entries.map(item => ({
      key: item.key,
      label: item.label,
      value: Number(item.value.toFixed(2)),
      weight: Number(item.weight.toFixed(2)),
    })),
    metrics: keys,
  };
}

function getActiveGlobalMetricKeys() {
  const keys = normalizeMetricKeys(state.selectedGlobalMetrics);
  state.selectedGlobalMetrics = [...keys];
  return keys;
}

function applyGlobalScoreToRow(row, metricKeys = getActiveGlobalMetricKeys()) {
  if (!row) return row;
  const details = computeGlobalScore(row, metricKeys);
  if (details) {
    row.globalScore = details.score;
    row.globalScoreBreakdown = details.breakdown;
    row.globalScoreDetails = details;
  } else {
    delete row.globalScore;
    delete row.globalScoreBreakdown;
    delete row.globalScoreDetails;
  }
  return row;
}

function recomputeGlobalScores(metricKeys = getActiveGlobalMetricKeys()) {
  const keys = normalizeMetricKeys(metricKeys);
  state.selectedGlobalMetrics = [...keys];
  state.data = state.data.map(row => {
    const updated = { ...row };
    const sourceRef = getSourceReference(row);
    if (sourceRef !== undefined) {
      defineSourceReference(updated, sourceRef);
    }
    return applyGlobalScoreToRow(updated, keys);
  });
}

function getMetricsForUseCases(useCaseIds) {
  const metrics = new Set();
  useCaseIds.forEach(id => {
    const profile = getUseCaseWeightProfile(id);
    const profileMetrics = getMetricKeysFromProfile(profile);
    if (profileMetrics.length) {
      profileMetrics.forEach(metric => metrics.add(metric));
      return;
    }
    const config = USE_CASE_MAP.get(id);
    if (!config) return;
    config.metrics.forEach(metric => metrics.add(metric));
  });
  return metrics;
}

function formatMetricSummary(metrics, isDefault) {
  if (!useCaseSummary) return;
  useCaseSummary.setAttribute("role", "group");
  useCaseSummary.setAttribute("aria-live", "polite");
  if (!useCaseSummary.hasAttribute("aria-label")) {
    useCaseSummary.setAttribute("aria-label", "Répartition du score sélection");
  }
  const weights = getActiveMetricWeightMap();
  const entries = (Array.isArray(metrics) ? metrics : [])
    .map((key, index) => {
      const rawWeight = weights?.[key];
      const weight = typeof rawWeight === "number" && Number.isFinite(rawWeight)
        ? Math.max(rawWeight, 0)
        : 0;
      if (weight <= 0) {
        return null;
      }
      return {
        key,
        label: getMetricLabel(key),
        weight,
        color: getMetricColor(key, index),
      };
    })
    .filter(Boolean);

  const totalWeight = entries.reduce((acc, entry) => acc + entry.weight, 0);

  if (!entries.length || totalWeight <= 0) {
    useCaseSummary.hidden = true;
    useCaseSummary.textContent = "";
    delete useCaseSummary.dataset.state;
    return;
  }

  entries.forEach(entry => {
    entry.share = entry.weight / totalWeight;
  });

  useCaseSummary.hidden = false;
  useCaseSummary.dataset.state = isDefault ? "default" : "custom";
  useCaseSummary.textContent = "";

  const distribution = document.createElement("div");
  distribution.className = "metric-distribution";

  const caption = document.createElement("span");
  caption.className = "metric-distribution__caption";
  caption.textContent = isDefault
    ? "Pondération standard du score sélection"
    : "Pondération selon tes usages";
  distribution.appendChild(caption);

  const bar = document.createElement("div");
  bar.className = "metric-distribution__bar";
  const animatedSegments = [];

  entries.forEach(entry => {
    const segment = document.createElement("span");
    segment.className = "metric-distribution__segment";
    segment.dataset.metricKey = entry.key;
    segment.style.setProperty("--segment-color", entry.color);
    segment.style.setProperty("--segment-width", "0%");
    const share = entry.share * 100;
    const digits = share < 10 ? 1 : 0;
    segment.setAttribute(
      "aria-label",
      `${entry.label} : ${formatNumber(share, digits)} %`
    );
    animatedSegments.push({ element: segment, share });
    bar.appendChild(segment);
  });

  distribution.appendChild(bar);

  const legend = document.createElement("ul");
  legend.className = "metric-distribution__legend";

  entries.forEach(entry => {
    const legendItem = document.createElement("li");
    legendItem.className = "metric-distribution__legend-item";

    const swatch = document.createElement("span");
    swatch.className = "metric-distribution__legend-swatch";
    swatch.style.setProperty("--swatch-color", entry.color);

    const label = document.createElement("span");
    label.className = "metric-distribution__legend-label";

    const labelText = document.createElement("strong");
    labelText.textContent = entry.label;

    const share = entry.share * 100;
    const digits = share < 10 ? 1 : 0;
    const percent = document.createElement("span");
    percent.className = "metric-distribution__legend-percent";
    percent.textContent = `${formatNumber(share, digits)} %`;

    label.appendChild(labelText);
    label.appendChild(percent);

    legendItem.appendChild(swatch);
    legendItem.appendChild(label);
    legendItem.dataset.metricKey = entry.key;
    legend.appendChild(legendItem);
  });

  distribution.appendChild(legend);
  useCaseSummary.appendChild(distribution);

  window.requestAnimationFrame(() => {
    window.requestAnimationFrame(() => {
      animatedSegments.forEach(({ element, share }) => {
        element.style.setProperty("--segment-width", `${share}%`);
      });
    });
  });
}

function updateUseCaseSummary(metricKeys = getActiveGlobalMetricKeys()) {
  const keys = normalizeMetricKeys(metricKeys);
  const isDefault = !state.selectedUseCases || state.selectedUseCases.size === 0;
  formatMetricSummary(keys, isDefault);
}

function updateUseCaseUI() {
  if (!useCaseCheckboxes.length) return;
  const active = new Set(state.selectedUseCases);
  useCaseCheckboxes.forEach(input => {
    const id = input.value;
    const checked = active.has(id);
    input.checked = checked;
    const card = input.closest(".use-case-card");
    if (card) {
      card.dataset.checked = checked ? "true" : "false";
    }
  });
}

function handleUseCaseChange(event) {
  const checkbox = event.target;
  if (!checkbox || checkbox.type !== "checkbox") return;
  const id = checkbox.value;
  if (!id) return;
  if (checkbox.checked) {
    state.selectedUseCases.add(id);
  } else {
    state.selectedUseCases.delete(id);
  }
  invalidateMetricWeightCache();
  const metrics = state.selectedUseCases.size
    ? Array.from(getMetricsForUseCases(state.selectedUseCases))
    : [...DEFAULT_GLOBAL_METRIC_KEYS];
  recomputeGlobalScores(metrics);
  updateUseCaseSummary(metrics);
  updateUseCaseUI();
  updateSortButtons();
  render();
}

function renderUseCaseControls() {
  if (!useCaseGrid) return;
  const fragment = document.createDocumentFragment();
  USE_CASES.forEach(useCase => {
    const label = document.createElement("label");
    label.className = "use-case-card";
    label.dataset.useCaseId = useCase.id;

    const input = document.createElement("input");
    input.type = "checkbox";
    input.value = useCase.id;
    input.setAttribute("aria-label", useCase.label);
    input.addEventListener("change", handleUseCaseChange);

    const indicator = document.createElement("span");
    indicator.className = "use-case-indicator";
    indicator.setAttribute("aria-hidden", "true");

    const imageWrapper = document.createElement("span");
    imageWrapper.className = "use-case-image";
    const img = document.createElement("img");
    img.src = `icon/${useCase.image}`;
    img.alt = useCase.alt;
    imageWrapper.appendChild(img);

    const textWrapper = document.createElement("span");
    textWrapper.className = "use-case-text";
    const title = document.createElement("strong");
    title.textContent = useCase.label;
    textWrapper.appendChild(title);

    const meta = document.createElement("span");
    meta.className = "use-case-meta";
    const profileMetrics = getMetricKeysFromProfile(
      getUseCaseWeightProfile(useCase.id)
    );
    const metricCount = profileMetrics.length || useCase.metrics.length;
    meta.textContent = metricCount > 1
      ? `${metricCount} critères pris en compte`
      : `${metricCount} critère pris en compte`;
    meta.hidden = true;
    meta.setAttribute("aria-hidden", "true");
    textWrapper.appendChild(meta);

    label.append(input, indicator, imageWrapper, textWrapper);
    fragment.appendChild(label);
  });

  useCaseGrid.innerHTML = "";
  useCaseGrid.appendChild(fragment);
  useCaseCheckboxes = Array.from(useCaseGrid.querySelectorAll('input[type="checkbox"]'));
  updateUseCaseUI();
  updateUseCaseSummary(getActiveGlobalMetricKeys());
}

function resolvePerformanceMotorType(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "road_motor_type",
    "motor_type",
    "motorType",
    "motor_type_score",
    "onroad_motor_type",
    "offroad_motor_type",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const lookup = lookupComponentScore("Moteur", raw);
  if (lookup != null) return lookup;

  if (typeof raw === "string") {
    const label = normalizeLabel(raw);
    if (!label) return null;
    if (
      label.includes("premium") ||
      label.includes("performance line cx") ||
      label.includes("bosch cx") ||
      label.includes("yamaha pwx")
    ) {
      return 5;
    }
    if (label.includes("mid") && label.includes("drive")) {
      if (
        label.includes("bosch") ||
        label.includes("shimano") ||
        label.includes("steps") ||
        label.includes("brose") ||
        label.includes("yamaha")
      ) {
        return 5;
      }
      return 4;
    }
    if (label.includes("reduct") || label.includes("geared") || label.includes("planetary")) {
      return 3;
    }
    if (label.includes("direct")) {
      return 2;
    }
    if (label.includes("avant") || label.includes("front")) {
      return 1;
    }
    if (label.includes("rear") || label.includes("arriere")) {
      return 3;
    }
  }

  return null;
}

function resolvePerformanceMotorPower(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "road_motor_power",
    "motor_power",
    "motorPower",
    "power_nominal",
    "puissance_nominale",
    "motor_power_w",
    "motor_power_score",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const numeric = extractNumeric(raw);
  if (numeric == null) return null;

  if (numeric < 249) return 1;
  if (numeric <= 349) return 3;
  if (numeric <= 449) return 4;
  if (numeric <= 500) return 5;
  return 3;
}

function resolvePerformanceMotorTorque(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "road_motor_torque",
    "offroad_motor_torque",
    "motor_torque",
    "motor_torque_nm",
    "torque",
    "torque_nm",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const numeric = extractNumeric(raw);
  if (numeric == null) return null;

  if (numeric < 39) return 1;
  if (numeric <= 60) return 2;
  if (numeric <= 80) return 3;
  if (numeric <= 100) return 4;
  return 5;
}

function resolvePerformanceTransmission(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "road_transmission",
    "offroad_transmission",
    "transmission",
    "drivetrain",
    "derailleur_model",
    "derailleur_label",
    "transmission_label",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const lookup = lookupComponentScore("Derailleur", raw);
  if (lookup != null) return lookup;

  const speeds = getNumericValueFromSources(source, originalSource, [
    "derailleur_speeds",
    "speeds",
    "rear_speeds",
  ]);
  if (speeds != null) {
    return mapTransmissionSpeedsToScore(speeds);
  }

  return null;
}

function resolvePerformanceRoadTyres(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "road_tyres",
    "tyres_road",
    "pneus_road",
    "tyres_label",
    "tyres",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const lookup = lookupComponentScore("Pneus", raw);
  if (lookup != null) return lookup;

  function resolveWidth(value) {
    if (value == null) return null;
    if (Array.isArray(value)) {
      for (const item of value) {
        const width = resolveWidth(item);
        if (width != null) return width;
      }
      return null;
    }
    if (typeof value === "number") {
      return value;
    }

    const text = String(value).toLowerCase().replace(/,/g, ".");
    let match = text.match(/(\d+(?:\.\d+)?)\s*mm/);
    if (match) {
      return parseFloat(match[1]);
    }

    match = text.match(/x\s*(\d+(?:\.\d+)?)/);
    if (match) {
      const width = parseFloat(match[1]);
      if (!Number.isNaN(width)) {
        if (width > 10) return width;
        return width * 25.4;
      }
    }

    match = text.match(/(\d+(?:\.\d+)?)\s*(?:po|in|inch)/);
    if (match) {
      return parseFloat(match[1]) * 25.4;
    }

    const numbers = text.match(/\d+(?:\.\d+)?/g);
    if (numbers && numbers.length) {
      const candidate = parseFloat(numbers[numbers.length - 1]);
      if (!Number.isNaN(candidate)) {
        if (candidate <= 10) return candidate * 25.4;
        return candidate;
      }
    }

    return null;
  }

  const widthMm = resolveWidth(raw);
  if (widthMm != null) {
    if (widthMm >= 95) return 1;
    if (widthMm >= 70) return 2;
    if (widthMm >= 50) return 3;
    if (widthMm >= 35) return 4;
    return 5;
  }

  if (!raw) return null;
  const label = normalizeLabel(raw);
  if (!label) return null;
  if (label.includes("fat") || label.includes("plus")) return 1;
  if (label.includes("mtb") || label.includes("trail") || label.includes("crampon") || label.includes("knob")) {
    return 2;
  }
  if (label.includes("hybrid") || label.includes("gravel") || label.includes("multisurface")) {
    return 3;
  }
  if (label.includes("city") || label.includes("urbain") || label.includes("slick")) {
    return 4;
  }
  if (label.includes("route") || label.includes("road")) {
    return 5;
  }
  return null;
}

function resolvePerformanceGravelTyres(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "gravel_tyres",
    "tyres_gravel",
    "pneus_gravel",
    "tyres_label",
    "tyres",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const lookup = lookupComponentScore("gravel_performance-06", raw);
  if (lookup != null) return lookup;

  const values = [];
  if (raw !== undefined && raw !== null) {
    if (Array.isArray(raw)) {
      values.push(...raw);
    } else if (typeof raw === "object") {
      values.push(...Object.values(raw));
    } else {
      values.push(raw);
    }
  }

  const widthInches = extractTyreWidthInches(values);
  if (widthInches != null) {
    const widthMm = widthInches * 25.4;
    if (widthMm >= 85) return 1;
    if (widthMm >= 65) return 2;
    if (widthMm >= 50) return 4;
    if (widthMm >= 38) return 5;
    if (widthMm >= 32) return 3;
    return 2;
  }

  if (!raw) return null;
  const label = normalizeLabel(raw);
  if (!label) return null;

  if (
    label.includes("gravelking") ||
    label.includes("gravel king") ||
    label.includes("gravel") ||
    label.includes("rambler") ||
    label.includes("g one") ||
    label.includes("pathfinder") ||
    label.includes("terra speed") ||
    label.includes("terra trail") ||
    label.includes("gr1") ||
    label.includes("gr2")
  ) {
    return 5;
  }
  if (
    label.includes("allroad") ||
    label.includes("all road") ||
    label.includes("allround") ||
    label.includes("all-round") ||
    label.includes("mixed") ||
    label.includes("multi surface") ||
    label.includes("multi-surface") ||
    label.includes("semi slick") ||
    label.includes("semi-slick") ||
    label.includes("trekking") ||
    label.includes("adventure")
  ) {
    return 4;
  }
  if (
    label.includes("mtb") ||
    label.includes("trail") ||
    label.includes("enduro") ||
    label.includes("cross country") ||
    label.includes("xc") ||
    label.includes("knob")
  ) {
    return 3;
  }
  if (label.includes("fat") || label.includes("plus") || label.includes("studded") || label.includes("spike")) {
    return 2;
  }
  if (
    label.includes("slick") ||
    label.includes("road") ||
    label.includes("route") ||
    label.includes("city") ||
    label.includes("urban") ||
    label.includes("urbain")
  ) {
    return 1;
  }
  return null;
}

function resolvePerformanceWeight(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "weight",
    "poids",
    "bike_weight",
    "poids_kg",
    "weight_kg",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const lookup = lookupComponentScore("Poids_avec_batterie", raw);
  if (lookup != null) return lookup;

  const numeric = extractNumeric(raw);
  if (numeric == null) return null;

  if (numeric > 32) return 1;
  if (numeric > 27) return 2;
  if (numeric > 23) return 3;
  if (numeric >= 20) return 4;
  return 5;
}

function resolvePerformanceAerodynamics(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "aerodynamics",
    "frame_aero",
    "road_aero",
    "cadre_aero",
    "geometry_aero",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;
  if (!raw) return null;

  const label = normalizeLabel(raw);
  if (!label) return null;

  const mapping = new Map([
    [5, ["route", "road", "tt", "ville", "city", "urbain", "commuter"]],
    [4, ["gravel", "fitness", "trekking", "hybride", "hybrid", "cyclocross"]],
    [3, ["montagne", "mountain", "trail", "enduro", "all mountain"]],
    [2, ["cargo", "utilitaire", "cruiser"]],
    [1, ["fatbike", "fat bike", "snow"]],
  ]);

  for (const [score, keywords] of mapping.entries()) {
    if (keywords.some((keyword) => label.includes(keyword))) {
      return score;
    }
  }

  if (label.includes("aero")) return 5;
  return null;
}

function resolvePerformanceAssistanceSoftware(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "display",
    "screen",
    "console",
    "ecran",
    "assistance_logiciel",
    "assistance_logicielle",
    "assistance_software",
    "software_support",
    "motor_assistance",
    "assist_levels",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;
  const lookup = lookupComponentScore("Display", raw);
  if (lookup != null) return lookup;
  if (!raw) return null;

  const label = normalizeLabel(raw);
  if (!label) return null;
  if (label.includes("kiox") || label.includes("nyon") || label.includes("connecte")) return 5;
  if (label.includes("couleur") || label.includes("smart system") || label.includes("steps")) return 4;
  if (label.includes("bosch") || label.includes("yamaha") || label.includes("brose") || label.includes("shimano")) return 3;
  if (label.includes("bafang") || label.includes("lcd")) return 2;
  if (label.includes("basique") || label.includes("3 niveau") || label.includes("trois niveaux")) return 1;
  return null;
}

function resolvePerformanceThrottle(source, originalSource) {
  const category = "utility_comfort-05";
  const fallbackMax = 10;
  const maxScore = getComponentTableMaxScore(category, fallbackMax) || fallbackMax;
  const values = collectValuesFromSources(source, originalSource, [
    "throttle",
    "has_throttle",
    "accelerateur",
    "gas_handle",
  ]);

  const normalizedValues = values.map(value => {
    const interpreted = interpretBooleanLike(value);
    return interpreted !== null ? interpreted : value;
  });

  if (normalizedValues.length) {
    const lookup = getComponentScoreFromLabels(category, normalizedValues, maxScore);
    if (lookup != null) return lookup;

    for (const value of normalizedValues) {
      if (typeof value === "boolean") {
        return value ? maxScore : 0;
      }
    }
  }

  const raw = normalizedValues.length ? normalizedValues[0] : null;
  if (raw == null) return null;

  const rating = coerceDirectRating(raw, maxScore);
  if (rating != null) return rating;

  const interpreted = interpretBooleanLike(raw);
  if (interpreted != null) {
    return interpreted ? maxScore : 0;
  }

  if (typeof raw === "string") {
    const label = normalizeLabel(raw);
    if (!label) return null;
    if (BOOLEAN_TRUE_LABELS.has(label)) {
      return maxScore;
    }
    if (BOOLEAN_FALSE_LABELS.has(label)) {
      return 0;
    }
  }

  return null;
}

function resolvePerformanceMotorResponsiveness(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "motor_responsiveness",
    "responsiveness",
    "assistance_mode",
    "mtb_mode",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;
  if (!raw) return null;

  const label = normalizeLabel(raw);
  if (!label) return null;
  if (label.includes("e mtb") || label.includes("adaptatif") || label.includes("intelligent")) {
    return 5;
  }
  if (label.includes("mode trail") || label.includes("trail")) {
    return 4;
  }
  if (label.includes("couple") && label.includes("cadence")) {
    return 3;
  }
  if (label.includes("cadence") && label.includes("retard")) {
    return 1;
  }
  if (label.includes("cadence")) {
    return 2;
  }
  return null;
}

function resolvePerformanceSuspension(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "suspension",
    "suspension_type",
    "fork_travel",
    "suspension_travel",
    "rear_travel",
    "frame_suspension",
    "suspension_label",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  let lookup = lookupComponentScore("Suspension_avant", raw);
  if (lookup == null) {
    lookup = lookupComponentScore("Suspension_arriere", raw);
  }
  if (lookup != null) return lookup;

  if (raw == null) return null;

  if (typeof raw === "string") {
    const label = normalizeLabel(raw);
    if (!label) return null;
    if (label.includes("170") || label.includes("180") || label.includes("200") || label.includes("dh")) {
      return 5;
    }
    if (label.includes("160") || label.includes("150") || label.includes("enduro")) {
      return 4;
    }
    if (label.includes("120") || label.includes("full") || label.includes("double")) {
      return 3;
    }
    if (label.includes("semi") || label.includes("hardtail")) {
      return 2;
    }
    if (label.includes("rigide") || label.includes("rigid")) {
      return 1;
    }
  }

  const numeric = extractNumeric(raw);
  if (numeric == null) return null;
  if (numeric >= 170) return 5;
  if (numeric >= 140) return 4;
  if (numeric >= 120) return 3;
  if (numeric > 0) return 2;
  return 1;
}

function resolvePerformanceFrontSuspension(source, originalSource) {
  const values = collectValuesFromSources(source, originalSource, FRONT_SUSPENSION_FIELDS);
  if (!values.length) return null;

  if (values.some(isSuspensionValueAbsent)) {
    return 1;
  }

  const maxScore = getComponentTableMaxScore("offroad_performance-04");
  const lookup = getComponentScoreFromLabels("offroad_performance-04", values, maxScore);
  if (lookup != null) return lookup;

  for (const value of values) {
    const result = resolvePerformanceSuspension({ suspension: value }, {});
    if (result != null) {
      return clamp(result, 0, maxScore || 5);
    }
  }

  return null;
}

function resolvePerformanceRearSuspension(source, originalSource) {
  const values = collectValuesFromSources(source, originalSource, REAR_SUSPENSION_FIELDS);
  if (!values.length) return null;

  if (values.some(isSuspensionValueAbsent)) {
    return 1;
  }

  const maxScore = getComponentTableMaxScore("offroad_performance-05");
  const lookup = getComponentScoreFromLabels("offroad_performance-05", values, maxScore);
  if (lookup != null) return lookup;

  for (const value of values) {
    const result = resolvePerformanceSuspension({ suspension: value }, {});
    if (result != null) {
      return clamp(result, 0, maxScore || 5);
    }
  }

  return null;
}

function resolvePerformanceBrakes(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "brakes",
    "brake_system",
    "freins",
    "brakes_model",
    "brakes_label",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const lookup = lookupComponentScore("Freins", raw);
  if (lookup != null) return lookup;

  if (!raw) return null;
  const label = normalizeLabel(raw);
  if (!label) return null;
  if (label.includes("magura mt7") || label.includes("code ultimate") || label.includes("dh")) {
    return 5;
  }
  if (label.includes("slx") || label.includes("xt") || label.includes("4 piston") || label.includes("g2")) {
    return 4;
  }
  if (label.includes("deore") || label.includes("level") || label.includes("guide")) {
    return 3;
  }
  if (label.includes("hydraulique") || label.includes("mt200")) {
    return 2;
  }
  if (label.includes("mecanique") || label.includes("cable")) {
    return 1;
  }
  return null;
}

function resolvePerformanceOffroadTyres(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "offroad_tyres",
    "mtb_tyres",
    "tyres_mtb",
    "pneus_vtt",
    "tyres",
    "tyres_label",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const lookup = lookupComponentScore("Pneus", raw);
  if (lookup != null) return lookup;

  if (!raw) return null;
  const label = normalizeLabel(raw);
  if (!label) return null;
  if (label.includes("dh") || label.includes("fat") || label.includes("stud") || label.includes("max grip")) {
    return 5;
  }
  if (label.includes("maxxis minion") || label.includes("magic mary") || label.includes("enduro")) {
    return 4;
  }
  if (label.includes("xc") || label.includes("cross country") || label.includes("leger")) {
    return 3;
  }
  if (label.includes("hybrid") || label.includes("semi slick")) {
    return 2;
  }
  if (label.includes("slick")) {
    return 1;
  }
  return null;
}

function resolvePerformanceGeometry(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "geometry",
    "frame_geometry",
    "cadre_geometry",
    "geometry_score",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;
  if (!raw) return null;

  const label = normalizeLabel(raw);
  if (!label) return null;
  if (label.includes("dh") || label.includes("downhill") || label.includes("race")) {
    return 5;
  }
  if (label.includes("enduro") || label.includes("modern")) {
    return 4;
  }
  if (label.includes("trail") || label.includes("all mountain")) {
    return 3;
  }
  if (label.includes("gravel") || label.includes("rigide")) {
    return 2;
  }
  if (label.includes("urbain") || label.includes("city")) {
    return 1;
  }
  return null;
}

function resolvePerformancePosition(source, originalSource) {
  const category = "utility_comfort-01";
  const fallbackMax = 10;
  const maxScore = getComponentTableMaxScore(category, fallbackMax) || fallbackMax;
  const rawValues = collectValuesFromSources(source, originalSource, ERGONOMICS_FIELDS);
  const normalizedValues = rawValues.map(value => {
    const interpreted = interpretBooleanLike(value);
    return interpreted !== null ? interpreted : value;
  });

  if (!normalizedValues.length) {
    return null;
  }

  return getComponentScoreFromLabels(category, normalizedValues, maxScore);
}

function resolvePerformanceLighting(source, originalSource) {
  let raw = getFirstValueFromSources(source, originalSource, [
    "lighting",
    "lights",
    "lumiere",
    "eclairage",
  ]);

  if (typeof raw === "boolean") {
    return raw ? 2 : 0;
  }

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  if (raw == null) {
    const front = getBooleanValueFromSources(source, originalSource, ["lights_front"]);
    const rear = getBooleanValueFromSources(source, originalSource, ["lights_rear"]);
    if (front != null || rear != null) {
      const count = Number(Boolean(front)) + Number(Boolean(rear));
      if (count === 2) return 3;
      if (count === 1) return 2;
      return 0;
    }
    return null;
  }

  const label = normalizeLabel(raw);
  if (!label) return null;
  if (label.includes("premium") || label.includes("haut lumen") || label.includes("stop arriere")) {
    return 5;
  }
  if (label.includes("integre") && label.includes("puissant")) {
    return 4;
  }
  if (label.includes("integre")) {
    return 3;
  }
  if (label.includes("ajoute") || label.includes("usb") || label.includes("pile")) {
    return 2;
  }
  if (label.includes("aucun") || label.includes("sans")) {
    return 0;
  }
  return null;
}

function resolvePerformanceLightingFront(source, originalSource) {
  const category = "utility_comfort-02";
  const fallbackMax = 10;
  const maxScore = getComponentTableMaxScore(category, fallbackMax) || fallbackMax;
  const fields = [
    "lighting_front",
    "lights_front",
    "front_light",
    "front_lighting",
    "eclairage_avant",
    ...LIGHTING_FIELDS,
  ];
  const values = collectValuesFromSources(source, originalSource, fields);
  const normalizedValues = values.map(value => {
    const interpreted = interpretBooleanLike(value);
    return interpreted !== null ? interpreted : value;
  });

  if (!normalizedValues.length) {
    return null;
  }

  return getComponentScoreFromLabels(category, normalizedValues, maxScore);
}

function resolvePerformanceLightingRear(source, originalSource) {
  const category = "utility_comfort-06";
  const fallbackMax = 10;
  const maxScore = getComponentTableMaxScore(category, fallbackMax) || fallbackMax;
  const fields = [
    "lighting_rear",
    "lights_rear",
    "rear_light",
    "tail_light",
    "feu_arriere",
    "eclairage_arriere",
    ...LIGHTING_FIELDS,
  ];
  const values = collectValuesFromSources(source, originalSource, fields);
  const normalizedValues = values.map(value => {
    const interpreted = interpretBooleanLike(value);
    return interpreted !== null ? interpreted : value;
  });

  if (!normalizedValues.length) {
    return null;
  }

  return getComponentScoreFromLabels(category, normalizedValues, maxScore);
}

function resolvePerformanceFenders(source, originalSource) {
  const category = "utility_comfort-03";
  const fallbackMax = 10;
  const maxScore = getComponentTableMaxScore(category, fallbackMax) || fallbackMax;
  const values = collectValuesFromSources(source, originalSource, FENDERS_FIELDS);
  const normalizedValues = values.map(value => {
    const interpreted = interpretBooleanLike(value);
    return interpreted !== null ? interpreted : value;
  });

  if (!normalizedValues.length) {
    return null;
  }

  return getComponentScoreFromLabels(category, normalizedValues, maxScore);
}

function resolvePerformanceRack(source, originalSource) {
  let raw = getFirstValueFromSources(source, originalSource, [
    "rack",
    "porte_bagages",
    "rear_rack",
    "portage",
  ]);

  if (typeof raw === "boolean") {
    return raw ? 5 : 0;
  }

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  if (raw == null) {
    const rear = getBooleanValueFromSources(source, originalSource, ["rack_rear"]);
    const front = getBooleanValueFromSources(source, originalSource, ["rack_front"]);
    if (rear != null || front != null) {
      if (rear && front) return 5;
      if (rear || front) return 3;
      return 0;
    }
    return null;
  }

  const label = normalizeLabel(raw);
  if (!label) return null;
  if (
    label.includes("robuste") ||
    label.includes("integr") ||
    label.includes("30 kg") ||
    label.includes("25 kg")
  ) {
    return 5;
  }
  if (label.includes("option") || label.includes("compatible")) {
    return 2;
  }
  if (label.includes("absent") || label.includes("aucun")) {
    return 0;
  }
  return null;
}

function resolvePerformanceMechanicalComfort(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "mechanical_confort",
    "mechanical_comfort",
    "comfort_package",
    "confort_mecanique",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;
  if (!raw) return null;

  const label = normalizeLabel(raw);
  if (!label) return null;
  const hasBalloon =
    label.includes("ballon") || label.includes("balloon") || label.includes("large tyre") || label.includes("large tire");
  const hasSuspSeat =
    label.includes("tige") || label.includes("susp") || label.includes("selle suspendue") || label.includes("seatpost");
  const hasFork = label.includes("fourche") || label.includes("fork");

  if (label.includes("set complet") || (hasBalloon && hasSuspSeat && hasFork)) {
    return 5;
  }
  const count = [hasBalloon, hasSuspSeat, hasFork].filter(Boolean).length;
  if (count >= 2) return 4;
  if (hasSuspSeat || hasFork) return 3;
  if (hasBalloon) return 2;
  if (label.includes("aucun") || label.includes("rigide")) return 1;
  return null;
}

function resolvePerformanceAutonomyWh(source, originalSource) {
  const category = "private_performance-02";
  const maxScore = getComponentTableMaxScore(category);
  const values = collectValuesFromSources(source, originalSource, [
    "battery_cold_wh",
    "battery_wh_cold",
    "autonomy_wh",
    "autonomie_wh",
    "autonomie_totale_wh",
    "autonomy_total_wh",
    "batterie_cap_wh",
    "battery_wh",
    "batteryWh",
  ]);

  if (!values.length) {
    const fallback = resolveBatteryCapacity(source, originalSource);
    return fallback != null ? clamp(fallback, 0, maxScore || 5) : null;
  }

  const directLookup = lookupComponentScore(category, values);
  if (directLookup != null) return directLookup;

  for (const value of values) {
    const numeric = extractNumeric(value);
    if (numeric != null && numeric > 0) {
      const label = mapBatteryCapacityToLabel(numeric);
      if (label) {
        const score = getComponentScoreFromLabels(category, label, maxScore);
        if (score != null) return score;
      }
    }
  }

  const fallback = resolveBatteryCapacity(source, originalSource);
  return fallback != null ? clamp(fallback, 0, maxScore || 5) : null;
}

function resolvePerformanceAllTerrainTyres(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "tyres_allterrain",
    "allterrain_tyres",
    "pneus_allterrain",
    "tyres",
    "tyres_label",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const lookup = lookupComponentScore("private_performance-05", raw);
  if (lookup != null) return lookup;

  const values = [];
  if (raw !== undefined && raw !== null) {
    if (Array.isArray(raw)) {
      values.push(...raw);
    } else if (typeof raw === "object") {
      values.push(...Object.values(raw));
    } else {
      values.push(raw);
    }
  }

  const widthInches = extractTyreWidthInches(values);
  if (widthInches != null) {
    const widthMm = widthInches * 25.4;
    if (widthMm >= 95) return 5;
    if (widthMm >= 75) return 4;
    if (widthMm >= 55) return 3;
    if (widthMm >= 40) return 2;
    return 1;
  }

  if (!raw) return null;
  const label = normalizeLabel(raw);
  if (!label) return null;

  if (label.includes("spike") || label.includes("stud") || label.includes("clou")) {
    return 5;
  }
  if (
    label.includes("fat") ||
    label.includes("plus") ||
    label.includes("maxxis minion") ||
    label.includes("assegai") ||
    label.includes("mud") ||
    label.includes("bog")
  ) {
    return 4;
  }
  if (
    label.includes("all terrain") ||
    label.includes("all-terrain") ||
    label.includes("allround") ||
    label.includes("hybrid") ||
    label.includes("gravel") ||
    label.includes("trekking")
  ) {
    return 3;
  }
  if (label.includes("slick") || label.includes("road") || label.includes("route") || label.includes("city")) {
    return 1;
  }
  return 2;
}

function resolvePerformanceRackFront(source, originalSource) {
  const category = "utility_comfort-07";
  const fallbackMax = 10;
  const maxScore = getComponentTableMaxScore(category, fallbackMax) || fallbackMax;
  const values = collectValuesFromSources(source, originalSource, [
    "rack_front",
    "front_rack",
    "porte_bagages_avant",
    "porte_bagage_avant",
    "porte_bagages_front",
  ]);

  const normalizedValues = values.map(value => {
    const interpreted = interpretBooleanLike(value);
    return interpreted !== null ? interpreted : value;
  });

  if (normalizedValues.length) {
    const lookup = getComponentScoreFromLabels(category, normalizedValues, maxScore);
    if (lookup != null) return lookup;

    for (const value of normalizedValues) {
      if (typeof value === "boolean") {
        return value ? maxScore : 0;
      }
    }
  }

  if (!normalizedValues.length) {
    const value = getBooleanValueFromSources(source, originalSource, ["rack_front"]);
    if (value == null) return null;
    return value ? maxScore : 0;
  }

  for (const value of normalizedValues) {
    if (typeof value === "number") {
      if (value <= 0) return 0;
      if (value >= 25) return maxScore;
      if (value >= 15) return clamp((4 / 5) * maxScore, 0, maxScore);
      if (value >= 5) return clamp((3 / 5) * maxScore, 0, maxScore);
      return clamp((2 / 5) * maxScore, 0, maxScore);
    }
    if (typeof value === "string") {
      const label = normalizeLabel(value);
      if (!label) continue;
      if (
        label.includes("integr") ||
        label.includes("inclus") ||
        label.includes("built in") ||
        label.includes("porteur") ||
        label.includes("front rack")
      ) {
        return maxScore;
      }
      if (label.includes("plateforme") || label.includes("heavy duty") || label.includes("support")) {
        return clamp((4 / 5) * maxScore, 0, maxScore);
      }
      if (label.includes("compatible") || label.includes("bosses") || label.includes("mount")) {
        return clamp((3 / 5) * maxScore, 0, maxScore);
      }
      if (label.includes("option") || label.includes("kit") || label.includes("accessoire")) {
        return clamp((2 / 5) * maxScore, 0, maxScore);
      }
      if (label.includes("aucun") || label.includes("none") || label.includes("sans")) {
        return 0;
      }
    }
  }

  return clamp((3 / 5) * maxScore, 0, maxScore);
}

function resolvePerformanceRackRear(source, originalSource) {
  const category = "utility_comfort-04";
  const fallbackMax = 10;
  const maxScore = getComponentTableMaxScore(category, fallbackMax) || fallbackMax;
  const values = collectValuesFromSources(source, originalSource, [
    "rack_rear",
    "rear_rack",
    "porte_bagages",
    "porte_bagages_arriere",
    "porte_bagage_arriere",
  ]);

  const normalizedValues = values.map(value => {
    const interpreted = interpretBooleanLike(value);
    return interpreted !== null ? interpreted : value;
  });

  if (normalizedValues.length) {
    const lookup = getComponentScoreFromLabels(category, normalizedValues, maxScore);
    if (lookup != null) return lookup;

    for (const value of normalizedValues) {
      if (typeof value === "boolean") {
        return value ? maxScore : 0;
      }
    }
  }

  if (!normalizedValues.length) {
    const value = getBooleanValueFromSources(source, originalSource, ["rack_rear", "rear_rack"]);
    if (value == null) return null;
    return value ? maxScore : 0;
  }

  for (const value of normalizedValues) {
    if (typeof value === "number") {
      if (value <= 0) return 0;
      if (value >= 35) return maxScore;
      if (value >= 25) return clamp((4 / 5) * maxScore, 0, maxScore);
      if (value >= 15) return clamp((3 / 5) * maxScore, 0, maxScore);
      return clamp((2 / 5) * maxScore, 0, maxScore);
    }
    if (typeof value === "string") {
      const label = normalizeLabel(value);
      if (!label) continue;
      if (label.includes("integr") || label.includes("inclus") || label.includes("heavy duty") || label.includes("30 kg")) {
        return maxScore;
      }
      if (label.includes("porte bagages") || label.includes("rear rack") || label.includes("pannier")) {
        return clamp((4 / 5) * maxScore, 0, maxScore);
      }
      if (label.includes("compatible") || label.includes("bosses") || label.includes("mount")) {
        return clamp((3 / 5) * maxScore, 0, maxScore);
      }
      if (label.includes("option") || label.includes("kit") || label.includes("accessoire")) {
        return clamp((2 / 5) * maxScore, 0, maxScore);
      }
      if (label.includes("aucun") || label.includes("none") || label.includes("sans")) {
        return 0;
      }
    }
  }

  return clamp((3 / 5) * maxScore, 0, maxScore);
}

function resolvePerformanceLightsFront(source, originalSource) {
  const category = "private_performance-07";
  const values = collectValuesFromSources(source, originalSource, [
    "lights_front",
    "front_light",
    "front_lighting",
    "eclairage_avant",
    "lighting_front",
  ]);

  if (!values.length) {
    const value = getBooleanValueFromSources(source, originalSource, ["lights_front", "front_light"]);
    if (value == null) return null;
    return value ? 3 : 0;
  }

  const lookup = lookupComponentScore(category, values);
  if (lookup != null) return lookup;

  for (const value of values) {
    if (typeof value === "boolean") {
      return value ? 3 : 0;
    }
    if (typeof value === "number") {
      if (value <= 0) return 0;
      if (value >= 1200) return 5;
      if (value >= 800) return 4;
      if (value >= 300) return 3;
      return 2;
    }
    if (typeof value === "string") {
      const label = normalizeLabel(value);
      if (!label) continue;
      if (label.includes("haut lumen") || label.includes("1300") || label.includes("1200") || label.includes("longue portee")) {
        return 5;
      }
      if (label.includes("integr") || label.includes("dynamo") || label.includes("projecteur")) {
        return 4;
      }
      if (label.includes("led") || label.includes("inclu") || label.includes("integrated")) {
        return 3;
      }
      if (label.includes("usb") || label.includes("clip") || label.includes("accessoire")) {
        return 2;
      }
      if (label.includes("aucun") || label.includes("sans") || label.includes("none")) {
        return 0;
      }
      if (label.includes("option") || label.includes("compatible")) {
        return 1;
      }
    }
  }

  return 3;
}

function resolvePerformanceLightsRear(source, originalSource) {
  const category = "private_performance-08";
  const values = collectValuesFromSources(source, originalSource, [
    "lights_rear",
    "rear_light",
    "tail_light",
    "feu_arriere",
    "eclairage_arriere",
  ]);

  if (!values.length) {
    const value = getBooleanValueFromSources(source, originalSource, ["lights_rear", "rear_light", "tail_light"]);
    if (value == null) return null;
    return value ? 3 : 0;
  }

  const lookup = lookupComponentScore(category, values);
  if (lookup != null) return lookup;

  for (const value of values) {
    if (typeof value === "boolean") {
      return value ? 3 : 0;
    }
    if (typeof value === "string") {
      const label = normalizeLabel(value);
      if (!label) continue;
      if (label.includes("stop") || label.includes("feu stop") || label.includes("brake light")) {
        return 5;
      }
      if (label.includes("integr") || label.includes("inclus") || label.includes("led")) {
        return 4;
      }
      if (label.includes("clip") || label.includes("usb") || label.includes("accessoire")) {
        return 2;
      }
      if (label.includes("aucun") || label.includes("sans") || label.includes("none")) {
        return 0;
      }
      if (label.includes("option") || label.includes("compatible")) {
        return 1;
      }
    }
  }

  return 3;
}

function resolvePerformanceWinterTyres(source, originalSource) {
  const raw = getFirstValueFromSources(source, originalSource, [
    "tyres_winter",
    "winter_tyres",
    "pneus_hiver",
    "tyres",
    "tyres_label",
  ]);

  const rating = coerceDirectRating(raw, 5);
  if (rating != null) return rating;

  const lookup = lookupComponentScore("winter_performance-02", raw);
  if (lookup != null) return lookup;

  const values = [];
  if (raw !== undefined && raw !== null) {
    if (Array.isArray(raw)) {
      values.push(...raw);
    } else if (typeof raw === "object") {
      values.push(...Object.values(raw));
    } else {
      values.push(raw);
    }
  }

  const widthInches = extractTyreWidthInches(values);
  if (widthInches != null) {
    const widthMm = widthInches * 25.4;
    if (widthMm >= 95) return 5;
    if (widthMm >= 75) return 4;
    if (widthMm >= 55) return 3;
    if (widthMm >= 40) return 2;
    return 1;
  }

  if (!raw) return null;
  const label = normalizeLabel(raw);
  if (!label) return null;

  if (label.includes("stud") || label.includes("spike") || label.includes("clou")) {
    return 5;
  }
  if (label.includes("hiver") || label.includes("winter") || label.includes("neige") || label.includes("snow")) {
    return 4;
  }
  if (label.includes("fat") || label.includes("plus")) {
    return 4;
  }
  if (label.includes("all terrain") || label.includes("gravel")) {
    return 3;
  }
  if (label.includes("road") || label.includes("route") || label.includes("slick") || label.includes("city")) {
    return 1;
  }
  return 2;
}

function resolvePerformanceBatteryColdWh(source, originalSource) {
  const category = "winter_performance-04";
  const maxScore = getComponentTableMaxScore(category);
  const values = collectValuesFromSources(source, originalSource, [
    "battery_cold_wh",
    "battery_wh_cold",
    "autonomy_cold_wh",
    "autonomie_cold_wh",
    "battery_wh",
    "batteryWh",
  ]);

  if (!values.length) {
    const fallback = resolveBatteryCapacity(source, originalSource);
    return fallback != null ? clamp(fallback, 0, maxScore || 5) : null;
  }

  const directLookup = lookupComponentScore(category, values);
  if (directLookup != null) return directLookup;

  for (const value of values) {
    const numeric = extractNumeric(value);
    if (numeric != null && numeric > 0) {
      const label = mapBatteryCapacityToLabel(numeric);
      if (label) {
        const score = getComponentScoreFromLabels(category, label, maxScore);
        if (score != null) return score;
      }
    }
    if (typeof value === "string") {
      const label = normalizeLabel(value);
      if (label.includes("chauff") || label.includes("heated")) {
        return maxScore > 0 ? maxScore : 5;
      }
      if (label.includes("degrade") || label.includes("reduit")) {
        return clamp((maxScore || 5) * 0.4, 0, maxScore || 5);
      }
    }
  }

  const fallback = resolveBatteryCapacity(source, originalSource);
  return fallback != null ? clamp(fallback, 0, maxScore || 5) : null;
}

function resolvePerformanceFenderFront(source, originalSource) {
  const category = "winter_performance-05";
  const values = collectValuesFromSources(source, originalSource, [
    "fenders_front",
    "front_fender",
    "garde_boue_avant",
    "garde_boue_front",
  ]);

  if (!values.length) {
    const value = getBooleanValueFromSources(source, originalSource, ["fenders_front", "front_fender"]);
    if (value == null) return null;
    return value ? 5 : 0;
  }

  const lookup = lookupComponentScore(category, values);
  if (lookup != null) return lookup;

  for (const value of values) {
    if (typeof value === "boolean") {
      return value ? 5 : 0;
    }
    if (typeof value === "string") {
      const label = normalizeLabel(value);
      if (!label) continue;
      if (label.includes("integr") || label.includes("large") || label.includes("long")) {
        return 5;
      }
      if (label.includes("inclus") || label.includes("full coverage")) {
        return 4;
      }
      if (label.includes("compatible") || label.includes("bosses") || label.includes("mount")) {
        return 3;
      }
      if (label.includes("option") || label.includes("clip") || label.includes("accessoire")) {
        return 2;
      }
      if (label.includes("aucun") || label.includes("none") || label.includes("sans")) {
        return 0;
      }
    }
  }

  return 3;
}

function resolvePerformanceFenderRear(source, originalSource) {
  const category = "winter_performance-06";
  const values = collectValuesFromSources(source, originalSource, [
    "fenders_rear",
    "rear_fender",
    "garde_boue_arriere",
    "garde_boue_ar",
  ]);

  if (!values.length) {
    const value = getBooleanValueFromSources(source, originalSource, ["fenders_rear", "rear_fender"]);
    if (value == null) return null;
    return value ? 5 : 0;
  }

  const lookup = lookupComponentScore(category, values);
  if (lookup != null) return lookup;

  for (const value of values) {
    if (typeof value === "boolean") {
      return value ? 5 : 0;
    }
    if (typeof value === "string") {
      const label = normalizeLabel(value);
      if (!label) continue;
      if (label.includes("integr") || label.includes("long") || label.includes("full coverage")) {
        return 5;
      }
      if (label.includes("inclus") || label.includes("rear rack")) {
        return 4;
      }
      if (label.includes("compatible") || label.includes("bosses") || label.includes("mount")) {
        return 3;
      }
      if (label.includes("option") || label.includes("clip") || label.includes("accessoire")) {
        return 2;
      }
      if (label.includes("aucun") || label.includes("none") || label.includes("sans")) {
        return 0;
      }
    }
  }

  return 3;
}

function createBikeImageElement(row) {
  const wrapper = document.createElement("div");
  wrapper.className = "bike-photo";
  const imagePath = BIKE_IMAGE_MAP.get(row.id);
  const label = [row.brand, row.model].filter(Boolean).join(" ").trim();
  if (imagePath) {
    const img = document.createElement("img");
    img.src = imagePath;
    img.alt = label || "Photo du modèle";
    img.loading = "lazy";
    img.decoding = "async";
    wrapper.appendChild(img);
  } else {
    wrapper.classList.add("bike-photo--placeholder");
    const fallback = document.createElement("span");
    fallback.className = "bike-photo-placeholder";
    fallback.textContent = getBikeInitials(label);
    wrapper.appendChild(fallback);
  }
  if (label) {
    wrapper.title = label;
  }
  return wrapper;
}

function getBikeInitials(label) {
  if (!label) {
    return "?";
  }
  const tokens = label
    .split(/\s+/u)
    .filter(Boolean)
    .slice(0, 2);
  const initials = tokens
    .map(token => token.charAt(0))
    .join("")
    .toUpperCase();
  return initials || "?";
}

function updateActiveSortHeaders() {
  const activeKey = state.sortKey;
  const parentHeaders = new Set();

  sortableHeaderCells.forEach(headerCell => {
    const parent = headerCell.closest("th");
    if (parent) {
      parentHeaders.add(parent);
    }
    delete headerCell.dataset.activeSort;
  });

  parentHeaders.forEach(parent => {
    delete parent.dataset.activeSort;
  });

  sortableHeaderCells.forEach(headerCell => {
    if (!headerCell?.dataset?.scoreKey) return;
    if (headerCell.dataset.scoreKey === activeKey) {
      headerCell.dataset.activeSort = "true";
      const parent = headerCell.closest("th");
      if (parent) {
        parent.dataset.activeSort = "true";
      }
    }
  });
}

function render() {
  closeScorePopover();
  updateActiveSortHeaders();
  const rows = getFilteredRows();
  updateResultsCount(rows.length);
  if (!rows.length) {
    tableBody.innerHTML = '<tr><td colspan="5" class="empty">Aucun modèle ne correspond à la recherche.</td></tr>';
    return;
  }

  const fragment = document.createDocumentFragment();
  rows.forEach(row => {
    const tr = document.createElement("tr");
    tr.dataset.rowId = row.id;

    const brandCell = document.createElement("td");
    brandCell.dataset.label = "Marque";
    const brandWrapper = document.createElement("div");
    brandWrapper.className = "brand-cell";

    const brandName = document.createElement("span");
    brandName.className = "brand-name";
    brandName.textContent = row.brand;
    brandWrapper.appendChild(brandName);

    if (row.brandLegalName && row.brandLegalName.toLowerCase() !== row.brand.toLowerCase()) {
      const legal = document.createElement("span");
      legal.className = "brand-legal";
      legal.textContent = row.brandLegalName;
      brandWrapper.appendChild(legal);
    }

    if (row.brandFoundedYear || row.brandWebsite) {
      const meta = document.createElement("div");
      meta.className = "brand-meta";
      if (row.brandFoundedYear) {
        const badge = document.createElement("span");
        badge.className = "brand-founded";
        badge.textContent = `Depuis ${row.brandFoundedYear}`;
        meta.appendChild(badge);
      }
      if (row.brandWebsite) {
        const link = document.createElement("a");
        link.href = row.brandWebsite;
        link.target = "_blank";
        link.rel = "noreferrer";
        link.textContent = "Site officiel";
        meta.appendChild(link);
      }
      brandWrapper.appendChild(meta);
    }

    brandCell.appendChild(brandWrapper);
    tr.appendChild(brandCell);

    const modelCell = document.createElement("td");
    modelCell.dataset.label = "Modèle";
    const modelWrapper = document.createElement("div");
    modelWrapper.className = "model-cell";
    modelWrapper.appendChild(createBikeImageElement(row));

    const modelInfo = document.createElement("div");
    modelInfo.className = "model-info";

    const modelName = document.createElement("span");
    modelName.className = "model-name";
    modelName.textContent = row.model;
    modelInfo.appendChild(modelName);

    if (row.url) {
      const link = document.createElement("a");
      link.href = row.url;
      link.target = "_blank";
      link.rel = "noreferrer";
      link.textContent = "fiche";
      modelInfo.appendChild(link);
    }

    modelWrapper.appendChild(modelInfo);
    modelCell.appendChild(modelWrapper);

    const globalLabel = getMetricLabel("globalScore");
    const modelScoreCell = document.createElement("div");
    modelScoreCell.className = "score-cell model-score";
    modelScoreCell.dataset.label = globalLabel;
    modelScoreCell.dataset.section = "global";

    const modelScoreTitle = document.createElement("span");
    modelScoreTitle.className = "model-score-title";
    modelScoreTitle.textContent = globalLabel;
    modelScoreCell.appendChild(modelScoreTitle);
    modelScoreCell.appendChild(renderScoreBar(row.globalScore));

    decorateScoreCell(modelScoreCell, row, "globalScore");
    modelCell.appendChild(modelScoreCell);
    tr.appendChild(modelCell);

    const essentialsCell = document.createElement("td");
    essentialsCell.dataset.label = "Essentiels";
    essentialsCell.className = "core-stack-cell";
    essentialsCell.dataset.section = "core";

    const essentialsWrapper = document.createElement("div");
    essentialsWrapper.className = "core-stack";

    const coreMetrics = [
      { key: "fiabiliteScore", label: getMetricLabel("fiabiliteScore") },
      { key: "autonomieScore", label: getMetricLabel("autonomieScore") },
      { key: "utilityComfortScore", label: getMetricLabel("utilityComfortScore") },
    ];

    let hasActiveCoreSort = false;

    coreMetrics.forEach(({ key, label }) => {
      const metricCell = document.createElement("div");
      metricCell.className = "score-cell core-stack-item";
      metricCell.dataset.label = label;
      metricCell.dataset.section = "core";

      const metricTitle = document.createElement("span");
      metricTitle.className = "core-stack-item-title";
      metricTitle.textContent = label;
      metricCell.appendChild(metricTitle);

      metricCell.appendChild(renderScoreBar(row[key]));
      decorateScoreCell(metricCell, row, key);
      if (state.sortKey === key) {
        hasActiveCoreSort = true;
      }

      essentialsWrapper.appendChild(metricCell);
    });

    if (hasActiveCoreSort) {
      essentialsCell.dataset.activeSort = "true";
    } else {
      delete essentialsCell.dataset.activeSort;
    }

    essentialsCell.appendChild(essentialsWrapper);
    tr.appendChild(essentialsCell);

    const performanceGroups = [
      {
        title: "Efficacité & maîtrise",
        metrics: [
          { key: "roadPerformanceScore", label: getMetricLabel("roadPerformanceScore") },
          { key: "offroadPerformanceScore", label: getMetricLabel("offroadPerformanceScore") },
        ],
      },
      {
        title: "Exploration & endurance",
        metrics: [
          { key: "privatePerformanceScore", label: getMetricLabel("privatePerformanceScore") },
          { key: "winterPerformanceScore", label: getMetricLabel("winterPerformanceScore") },
        ],
      },
    ];

    performanceGroups.forEach(group => {
      const performanceCell = document.createElement("td");
      performanceCell.dataset.label = group.title;
      performanceCell.className = "performance-stack-cell";
      performanceCell.dataset.section = "performance";

      const performanceWrapper = document.createElement("div");
      performanceWrapper.className = "performance-stack";

      if (group.title) {
        const groupTitle = document.createElement("span");
        groupTitle.className = "performance-stack-title";
        groupTitle.textContent = group.title;
        performanceWrapper.appendChild(groupTitle);
      }

      let hasActivePerformanceSort = false;

      group.metrics.forEach(({ key, label }) => {
        const metricCell = document.createElement("div");
        metricCell.className = "score-cell performance-stack-item";
        metricCell.dataset.label = label;
        metricCell.dataset.section = "performance";

        const metricTitle = document.createElement("span");
        metricTitle.className = "performance-stack-item-title";
        metricTitle.textContent = label;
        metricCell.appendChild(metricTitle);

        metricCell.appendChild(renderScoreBar(row[key]));
        decorateScoreCell(metricCell, row, key);
        if (state.sortKey === key) {
          hasActivePerformanceSort = true;
        }

        performanceWrapper.appendChild(metricCell);
      });

      if (hasActivePerformanceSort) {
        performanceCell.dataset.activeSort = "true";
      } else {
        delete performanceCell.dataset.activeSort;
      }

      performanceCell.appendChild(performanceWrapper);
      tr.appendChild(performanceCell);
    });

    fragment.appendChild(tr);
  });

  tableBody.innerHTML = "";
  tableBody.appendChild(fragment);
}

function roundScoreValue(value, digits = 1) {
  if (typeof value !== "number" || !Number.isFinite(value)) return null;
  const precision = Math.max(0, Math.trunc(digits));
  const factor = 10 ** precision;
  return Math.round(value * factor) / factor;
}

function renderScoreBar(value) {
  const numericValue =
    typeof value === "number" && Number.isFinite(value)
      ? value
      : typeof value === "string"
        ? parseNumericString(value)
        : null;
  const roundedValue =
    numericValue == null ? null : roundScoreValue(numericValue, 1);

  const wrapper = document.createElement("div");
  wrapper.className = "score-bar";
  const label = document.createElement("div");
  label.className = "value";
  label.textContent =
    roundedValue == null ? "–" : roundedValue.toFixed(1);
  const bar = document.createElement("div");
  bar.className = "bar";
  const pct =
    roundedValue == null ? 0 : Math.round((clamp(roundedValue, 0, 10) / 10) * 100);
  bar.style.setProperty("--fill", `${pct}%`);
  const levelValue = roundedValue;
  wrapper.dataset.level = levelValue == null
    ? "none"
    : levelValue >= 8
      ? "high"
      : levelValue >= 6
        ? "mid"
        : levelValue >= 4
          ? "warn"
          : "low";
  wrapper.append(label, bar);
  return wrapper;
}

function decorateScoreCell(cell, row, scoreKey) {
  if (!cell) return;
  cell.dataset.scoreKey = scoreKey;
  if (state.sortKey === scoreKey) {
    cell.dataset.activeSort = "true";
  } else {
    delete cell.dataset.activeSort;
  }
  const details = getScoreDetails(row, scoreKey);
  if (details?.breakdown && details.breakdown.length) {
    cell.dataset.hasBreakdown = "true";
    cell.tabIndex = 0;
  } else {
    delete cell.dataset.hasBreakdown;
    cell.removeAttribute("tabindex");
  }
}

function getScoreDetails(row, scoreKey) {
  if (!row || !scoreKey) return null;
  const resolver = SCORE_DETAIL_RESOLVERS[scoreKey];
  if (typeof resolver === "function") {
    const resolved = resolver(row);
    if (resolved) {
      return resolved;
    }
  }

  const fallbackKey = SCORE_BREAKDOWN_KEYS[scoreKey];
  if (!fallbackKey) return null;
  const raw = row[fallbackKey];
  if (!raw) return null;

  let breakdown;
  if (Array.isArray(raw)) {
    breakdown = raw;
  } else if (typeof raw === "object") {
    breakdown = Object.entries(raw).map(([key, value]) => ({
      key,
      label: key,
      contribution: typeof value === "number" ? value : null,
    }));
  }

  if (!breakdown || breakdown.length === 0) return null;
  return {
    score: typeof row[scoreKey] === "number" ? Number(row[scoreKey]) : null,
    breakdown,
  };
}

function formatRawDisplay(value) {
  if (value === undefined || value === null) return null;
  if (Array.isArray(value)) {
    const parts = value.map(part => formatRawDisplay(part)).filter(Boolean);
    if (!parts.length) return null;
    return Array.from(new Set(parts)).join(" · ");
  }
  if (typeof value === "object") {
    return formatRawDisplay(Object.values(value));
  }
  if (typeof value === "boolean") {
    return value ? "Oui" : "Non";
  }
  if (typeof value === "number") {
    const digits = Number.isInteger(value) ? 0 : 1;
    return formatNumber(value, digits);
  }
  const text = String(value).trim();
  return text || null;
}

const createValueListResolver = keys => (row, original) => {
  if (!Array.isArray(keys) || !keys.length) return null;
  return collectValuesFromSources(original, row, keys);
};

const createStringResolver = keys => (row, original) => {
  if (!Array.isArray(keys) || !keys.length) return null;
  return getFirstValueFromSources(original, row, keys);
};

const createBooleanResolver = keys => (row, original) => {
  if (!Array.isArray(keys) || !keys.length) return null;
  const flag = getBooleanValueFromSources(original, row, keys);
  if (flag != null) {
    return flag ? "Oui" : "Non";
  }
  return getFirstValueFromSources(original, row, keys);
};

const MOTOR_FIELDS = [
  "motor_label",
  "motor_brand",
  "motor_model",
  "motor",
  "motorBrand",
  "motor_name",
  "road_motor_type",
  "offroad_motor_type",
];
const CONTROLLER_FIELDS = ["controller", "controller_label", "controleur"];
const DISPLAY_FIELDS = ["display", "screen", "console", "afficheur"];
const DERAILLEUR_FIELDS = [
  "derailleur_label",
  "derailleur",
  "transmission",
  "drivetrain",
  "transmission_label",
];
const BRAKES_FIELDS = ["brakes_label", "brakes", "freins", "brakes_model"];
const TYRES_FIELDS = [
  "tyres_label",
  "tyres",
  "pneus",
  "pneus_type",
  "pneus_vtt",
  "tyres_road",
  "offroad_tyres",
  "tyres_mtb",
];
const TRANSMISSION_FIELDS = [
  "transmission",
  "drivetrain",
  "derailleur_label",
  "transmission_label",
  "derailleur",
  "road_transmission",
  "offroad_transmission",
];
const MOTOR_TYPE_FIELDS = [
  "motor_type",
  "motorType",
  "moteur_type_label",
  "motor_label",
  "motor_category",
];
const MOTOR_POWER_NUMERIC_FIELDS = [
  "motor_power_w",
  "motorPowerW",
  "puissance_w",
  "motor_nominal_power",
  "power_nominal",
  "power_continuous",
];
const MOTOR_POWER_STRING_FIELDS = [
  "motor_power",
  "motorPower",
  "puissance",
  "motor_power_label",
];
const MOTOR_TORQUE_FIELDS = [
  "motor_torque_nm",
  "torque_nm",
  "motor_torque",
  "road_motor_torque",
  "offroad_motor_torque",
];
const FRAME_FIELDS = ["frame_suspension", "cadre_type", "frame_type", "cadre_susp"];
const TYRES_ROAD_FIELDS = ["road_tyres", "tyres_road", "pneus_road", "tyres_label", "tyres"];
const TYRES_OFFROAD_FIELDS = [
  "offroad_tyres",
  "mtb_tyres",
  "tyres_mtb",
  "pneus_vtt",
  "tyres_label",
  "tyres",
];
const SUSPENSION_FIELDS = [
  "suspension",
  "suspension_type",
  "fork_travel",
  "suspension_travel",
  "rear_travel",
  "frame_suspension",
  "suspension_label",
  "suspension_avant",
  "suspension_arriere",
];
const GEOMETRY_FIELDS = ["geometry", "frame_geometry", "cadre_geometry"];
const AERODYNAMICS_FIELDS = [
  "aerodynamics",
  "frame_aero",
  "road_aero",
  "cadre_aero",
  "geometry_aero",
];
const SOFTWARE_FIELDS = [
  "assistance_logiciel",
  "assistance_logicielle",
  "assistance_software",
  "software_support",
  "motor_assistance",
  "assist_levels",
];
const ERGONOMICS_FIELDS = ["position", "ergonomie", "ergonomics", "rider_position", "posture"];
const MECHANICAL_COMFORT_FIELDS = [
  "mechanical_confort",
  "mechanical_comfort",
  "comfort_package",
  "confort_mecanique",
];
const RANGE_EXTENDER_FIELDS = [
  "range_extender",
  "rangeExtender",
  "battery_addon",
  "battery_add_on",
  "bonus_add_on",
  "bonus_addon",
  "bonus_add-on",
  "bonusAddon",
];
const BATTERY_CELLS_FIELDS = ["battery_cells", "batteryCells", "cellules", "cellules_type"];
const FRONT_SUSPENSION_FIELDS = ["suspension_avant", "front_suspension"];
const REAR_SUSPENSION_FIELDS = ["suspension_arriere", "rear_suspension"];
const LIGHTING_FIELDS = ["lighting", "lights", "lumiere", "eclairage"];
const FENDERS_FIELDS = ["fenders", "garde_boue", "mudguards"];
const RACK_FIELDS = ["rack", "porte_bagages", "rear_rack", "portage"];
const MOTOR_RESPONSIVENESS_FIELDS = [
  "motor_responsiveness",
  "responsiveness",
  "assistance_mode",
  "mtb_mode",
];

function resolveMotorPowerValue(row, original) {
  const value = getNumericValueFromSources(original, row, MOTOR_POWER_NUMERIC_FIELDS);
  if (value != null) {
    const digits = Number.isInteger(value) ? 0 : 1;
    const formatted = formatNumber(value, digits);
    if (Math.abs(value) > 10) {
      return `${formatted} W`;
    }
    return formatted;
  }
  return getFirstValueFromSources(original, row, MOTOR_POWER_STRING_FIELDS);
}

function resolveMotorTorqueValue(row, original) {
  const value = getNumericValueFromSources(original, row, MOTOR_TORQUE_FIELDS);
  if (value != null) {
    const digits = Number.isInteger(value) ? 0 : 1;
    const formatted = formatNumber(value, digits);
    if (Math.abs(value) > 10) {
      return `${formatted} Nm`;
    }
    return formatted;
  }
  return getFirstValueFromSources(original, row, MOTOR_TORQUE_FIELDS);
}

function resolveRangeExtenderValue(row, original) {
  const flag = getBooleanValueFromSources(original, row, RANGE_EXTENDER_FIELDS);
  if (flag != null) {
    return flag ? "Oui" : "Non";
  }
  return getFirstValueFromSources(original, row, RANGE_EXTENDER_FIELDS);
}

function resolveThrottleValue(row, original) {
  const flag = getBooleanValueFromSources(original, row, [
    "throttle",
    "has_throttle",
    "accelerateur",
    "gas_handle",
  ]);
  if (flag != null) {
    return flag ? "Oui" : "Non";
  }
  return getFirstValueFromSources(original, row, [
    "throttle",
    "has_throttle",
    "accelerateur",
    "gas_handle",
  ]);
}

function resolveLightingValue(row, original) {
  const direct = getFirstValueFromSources(original, row, LIGHTING_FIELDS);
  if (typeof direct === "boolean") {
    return direct ? "Oui" : "Non";
  }
  const front = getBooleanValueFromSources(original, row, ["lights_front"]);
  const rear = getBooleanValueFromSources(original, row, ["lights_rear"]);
  if (front != null || rear != null) {
    if (front && rear) return "Avant + arrière";
    if (front) return "Avant";
    if (rear) return "Arrière";
    return "Aucun";
  }
  return direct;
}

function resolveFendersValue(row, original) {
  const directFlag = getBooleanValueFromSources(original, row, FENDERS_FIELDS);
  if (directFlag != null) {
    return directFlag ? "Oui" : "Non";
  }
  const direct = getFirstValueFromSources(original, row, FENDERS_FIELDS);
  const front = getBooleanValueFromSources(original, row, ["fenders_front"]);
  const rear = getBooleanValueFromSources(original, row, ["fenders_rear"]);
  if (front != null || rear != null) {
    if (front && rear) return "Avant + arrière";
    if (front) return "Avant";
    if (rear) return "Arrière";
    return "Aucun";
  }
  return direct;
}

function resolveRackValue(row, original) {
  const directFlag = getBooleanValueFromSources(original, row, RACK_FIELDS);
  if (directFlag != null) {
    return directFlag ? "Oui" : "Non";
  }
  const direct = getFirstValueFromSources(original, row, RACK_FIELDS);
  const rear = getBooleanValueFromSources(original, row, ["rack_rear"]);
  const front = getBooleanValueFromSources(original, row, ["rack_front"]);
  if (front != null || rear != null) {
    if (front && rear) return "Avant + arrière";
    if (rear) return "Arrière";
    if (front) return "Avant";
    return "Aucun";
  }
  return direct;
}

const BREAKDOWN_VALUE_RESOLVERS = {
  motor: createValueListResolver(MOTOR_FIELDS),
  controller: createValueListResolver(CONTROLLER_FIELDS),
  display: createValueListResolver(DISPLAY_FIELDS),
  derailleur: createValueListResolver(DERAILLEUR_FIELDS),
  brakes: createValueListResolver(BRAKES_FIELDS),
  tyres: createValueListResolver(TYRES_FIELDS),
  batteryCapacity: (row, original) => {
    const capacity = getNumericValueFromSources(original, row, [
      "batterie_cap_wh",
      "batteryWh",
      "battery_wh",
      "batteryCapacityWh",
      "battery_capacity_wh",
      "battery_capacity",
    ]);
    if (capacity != null) {
      return `${formatNumber(capacity, 0)} Wh`;
    }
    return getStringValueFromSources(original, row, [
      "battery_capacity_label",
      "battery_capacity_range",
    ]);
  },
  batteryCertification: (row, original) => {
    const label = getStringValueFromSources(original, row, [
      "battery_certification",
      "batterie_certification",
      "battery_certif",
    ]);
    if (label) return label;
    const ul = getBooleanValueFromSources(original, row, [
      "battery_certified_ul",
      "batterie_certif_ul",
    ]);
    if (ul === true) return "UL 2849";
    if (ul === false) return "Sans certif.";
    return null;
  },
  batteryCells: createValueListResolver(BATTERY_CELLS_FIELDS),
  weight: (row, original) => {
    const value = getNumericValueFromSources(original, row, [
      "weight_kg",
      "poids_kg",
      "bike_weight",
      "poids",
    ]);
    if (value != null) {
      const digits = Number.isInteger(value) ? 0 : 1;
      return `${formatNumber(value, digits)} kg`;
    }
    return getStringValueFromSources(original, row, ["weight_label", "poids_label"]);
  },
  stem: createBooleanResolver(["stem_adjustable", "potence_ajustable", "stem"]),
  frontSuspension: createValueListResolver(FRONT_SUSPENSION_FIELDS),
  rearSuspension: createValueListResolver(REAR_SUSPENSION_FIELDS),
  rangeExtender: resolveRangeExtenderValue,
  motorType: createValueListResolver(MOTOR_TYPE_FIELDS),
  motorPower: resolveMotorPowerValue,
  frameEfficiency: createValueListResolver(FRAME_FIELDS),
  tyres_road: createValueListResolver(TYRES_ROAD_FIELDS),
  tyres_offroad: createValueListResolver(TYRES_OFFROAD_FIELDS),
  transmission: createValueListResolver(TRANSMISSION_FIELDS),
  motor_type: createValueListResolver(MOTOR_TYPE_FIELDS),
  motor_power: resolveMotorPowerValue,
  motor_torque: resolveMotorTorqueValue,
  motor_responsiveness: createStringResolver(MOTOR_RESPONSIVENESS_FIELDS),
  suspension: createValueListResolver(SUSPENSION_FIELDS),
  geometry: createStringResolver(GEOMETRY_FIELDS),
  aerodynamics: createStringResolver(AERODYNAMICS_FIELDS),
  software: createStringResolver(SOFTWARE_FIELDS),
  throttle: resolveThrottleValue,
  lighting: resolveLightingValue,
  fenders: resolveFendersValue,
  rack: resolveRackValue,
  mechanical_comfort: createStringResolver(MECHANICAL_COMFORT_FIELDS),
  ergonomics: createStringResolver(ERGONOMICS_FIELDS),
};

function resolveBreakdownSourceValue(row, scoreKey, item) {
  if (!item) return null;
  const original = getSourceReference(row) ?? row;
  const resolver = item.key ? BREAKDOWN_VALUE_RESOLVERS[item.key] : null;
  if (typeof resolver === "function") {
    const resolved = resolver(row, original, scoreKey, item);
    const formatted = formatRawDisplay(resolved);
    if (formatted) return formatted;
  }
  if (Object.prototype.hasOwnProperty.call(item, "rawValue")) {
    const formatted = formatRawDisplay(item.rawValue);
    if (formatted) return formatted;
  }
  if (typeof item.value === "number" && Number.isFinite(item.value)) {
    const digits = Number.isInteger(item.value) ? 0 : 1;
    const formatted = formatNumber(item.value, digits);
    return scoreKey === "globalScore" ? `${formatted} / 10` : formatted;
  }
  if (item.key) {
    if (original && Object.prototype.hasOwnProperty.call(original, item.key)) {
      const formattedOriginal = formatRawDisplay(original[item.key]);
      if (formattedOriginal) return formattedOriginal;
    }
    if (row && Object.prototype.hasOwnProperty.call(row, item.key)) {
      const formattedRow = formatRawDisplay(row[item.key]);
      if (formattedRow) return formattedRow;
    }
  }
  if (typeof item.contribution === "string" && item.contribution.trim()) {
    return item.contribution.trim();
  }
  return null;
}

function openScorePopover(row, scoreKey, cell) {
  if (!scorePopover || !scorePopoverTitle || !scorePopoverBody) return;
  const label = cell?.dataset?.label ?? "Score";
  const rawValue = row?.[scoreKey];
  const numericValue = typeof rawValue === "number" && Number.isFinite(rawValue) ? rawValue : null;
  const scoreLabel = numericValue == null ? "–" : formatNumber(numericValue, 1);
  const scoreStars =
    numericValue == null ? null : formatStarRating(numericValue, 10);

  scorePopoverTitle.textContent = `${label} — ${scoreLabel} / 10`;
  if (scoreStars) {
    scorePopoverTitle.setAttribute(
      "aria-label",
      `${label} — ${scoreStars.label} (${scoreLabel} sur 10)`
    );
  } else {
    scorePopoverTitle.removeAttribute("aria-label");
  }
  scorePopoverBody.innerHTML = "";

  const details = getScoreDetails(row, scoreKey);
  if (details?.breakdown && details.breakdown.length) {
    const list = document.createElement("ul");
    list.className = "score-breakdown";

    const requirementHint = formatRequirementMessage(scoreKey);

    details.breakdown.forEach(item => {
      const li = document.createElement("li");
      li.className = "score-breakdown-item";
      const title = document.createElement("span");
      title.className = "score-breakdown-label";
      const labelText = item.label ?? item.key ?? "Critère";
      const sourceValue = resolveBreakdownSourceValue(row, scoreKey, item);
      title.textContent = sourceValue ? `${labelText}: ${sourceValue}` : labelText;
      const meta = document.createElement("span");
      meta.className = "score-breakdown-value";

      const fragments = [];
      const ariaParts = [];
      if (typeof item.score === "number" && Number.isFinite(item.score)) {
        const scoreMax =
          typeof item.maxScore === "number" && Number.isFinite(item.maxScore)
            ? item.maxScore
            : 5;
        const scoreStars = formatStarRating(item.score, scoreMax);
        if (scoreStars) {
          const starsElement = createStarElement(scoreStars);
          fragments.push({ type: "node", value: starsElement });
          ariaParts.push(
            `${scoreStars.label} (score ${formatNumber(
              item.score,
              Number.isInteger(item.score) ? 0 : 2
            )} sur ${formatNumber(
              scoreMax,
              Number.isInteger(scoreMax) ? 0 : 2
            )})`
          );
        }
      }
      if (typeof item.weight === "number" && Number.isFinite(item.weight)) {
        const digits = Number.isInteger(item.weight) ? 0 : 2;
        const weightText = `multiplicateur d'importance ×${formatNumber(
          item.weight,
          digits
        )}`;
        fragments.push({ type: "text", value: weightText });
        ariaParts.push(weightText);
      }
      if (typeof item.contribution === "number" && Number.isFinite(item.contribution)) {
        fragments.push({ type: "text", value: `→ +${formatNumber(item.contribution, 2)} pts` });
        ariaParts.push(`→ +${formatNumber(item.contribution, 2)} points`);
      } else if (typeof item.weighted === "number" && Number.isFinite(item.weighted)) {
        fragments.push({
          type: "text",
          value: `→ ${formatNumber(item.weighted, 2)} pts pondérés`,
        });
        ariaParts.push(`→ ${formatNumber(item.weighted, 2)} points pondérés`);
      } else if (typeof item.value === "number" && Number.isFinite(item.value)) {
        fragments.push({ type: "text", value: formatNumber(item.value, 2) });
        ariaParts.push(`valeur ${formatNumber(item.value, 2)}`);
      } else if (typeof item.contribution === "string") {
        fragments.push({ type: "text", value: item.contribution });
        ariaParts.push(item.contribution);
      }

      if (!fragments.length) {
        const fallback = requirementHint ?? "Détail indisponible";
        fragments.push({ type: "text", value: fallback });
        ariaParts.push(fallback);
      }

      meta.textContent = "";
      fragments.forEach((fragment, index) => {
        if (index > 0) {
          meta.append(document.createTextNode(" · "));
        }
        if (fragment.type === "node" && fragment.value instanceof Node) {
          meta.append(fragment.value);
        } else if (fragment.type === "text") {
          meta.append(document.createTextNode(fragment.value));
        }
      });
      if (ariaParts.length) {
        meta.setAttribute("aria-label", ariaParts.join(" · "));
      } else {
        meta.removeAttribute("aria-label");
      }
      li.append(title, meta);
      list.appendChild(li);
    });

    scorePopoverBody.appendChild(list);

    const summaryParts = [];
    const totalWeighted = typeof details.totalWeighted === "number" ? details.totalWeighted : null;
    const weightSum = typeof details.weightSum === "number" ? details.weightSum : null;
    const finalScore = typeof details.score === "number" ? details.score : numericValue;

    if (totalWeighted != null && weightSum != null) {
      summaryParts.push(
        `Somme pondérée : ${formatNumber(totalWeighted, 2)} / ${formatNumber(5 * weightSum, 2)}`
      );
    }
    if (finalScore != null) {
      summaryParts.push(`Score final : ${formatNumber(finalScore, 1)} / 10`);
    }

    if (summaryParts.length) {
      const summary = document.createElement("p");
      summary.className = "score-breakdown-summary";
      summary.textContent = summaryParts.join(" • ");
      scorePopoverBody.appendChild(summary);
    }
  } else {
    const message = document.createElement("p");
    message.className = "score-breakdown-empty";
    message.textContent =
      formatRequirementMessage(scoreKey) ?? "Aucun détail disponible pour ce score.";
    scorePopoverBody.appendChild(message);
  }

  scorePopover.dataset.open = "true";
  scorePopover.setAttribute("aria-hidden", "false");
  if (scorePopoverBody) {
    scorePopoverBody.scrollTop = 0;
  }
  activeScoreCell = cell ?? null;
  const card = scorePopover.querySelector(".score-popover-card");
  if (card) {
    card.focus();
  }
}

function closeScorePopover() {
  if (!scorePopover) return;
  scorePopover.dataset.open = "false";
  scorePopover.setAttribute("aria-hidden", "true");
  if (
    activeScoreCell &&
    typeof activeScoreCell.focus === "function" &&
    document.contains(activeScoreCell)
  ) {
    activeScoreCell.focus();
  }
  activeScoreCell = null;
}

function formatStarRating(value, maxValue, totalStars = 5) {
  if (
    typeof value !== "number" ||
    !Number.isFinite(value) ||
    typeof maxValue !== "number" ||
    !Number.isFinite(maxValue) ||
    maxValue <= 0 ||
    totalStars <= 0
  ) {
    return null;
  }
  const ratio = Math.max(0, Math.min(value / maxValue, 1));
  const rawRating = ratio * totalStars;
  const roundedRating = Math.round(rawRating * 10) / 10;
  const digits = Number.isInteger(roundedRating) ? 0 : 1;
  const labelValue = roundedRating.toLocaleString("fr-FR", {
    minimumFractionDigits: digits,
    maximumFractionDigits: digits,
  });
  const stars = "★".repeat(totalStars);
  return {
    stars,
    percent: Math.max(0, Math.min((roundedRating / totalStars) * 100, 100)),
    rating: roundedRating,
    totalStars,
    label: `${labelValue} ${roundedRating > 1 ? "étoiles" : "étoile"}`,
  };
}

function createStarElement(starInfo) {
  const span = document.createElement("span");
  span.className = "score-stars";
  span.setAttribute("aria-hidden", "true");
  span.textContent = starInfo.stars;
  span.dataset.stars = starInfo.stars;
  span.style.setProperty("--score-star-percent", `${starInfo.percent}%`);
  return span;
}

function formatList(values) {
  if (!Array.isArray(values) || values.length === 0) return "";
  if (values.length === 1) return values[0];
  if (values.length === 2) return `${values[0]} et ${values[1]}`;
  return `${values.slice(0, -1).join(", ")} et ${values.slice(-1)[0]}`;
}

function formatNumber(value, digits = 1) {
  if (typeof value !== "number" || !Number.isFinite(value)) return "–";
  return value.toLocaleString("fr-FR", {
    minimumFractionDigits: digits,
    maximumFractionDigits: digits,
  });
}

function getFilteredRows() {
  const query = state.query.trim().toLowerCase();
  let rows = state.data.filter(row => {
    const haystack = `${row.brand} ${row.model} ${row.brandLegalName ?? ""}`.toLowerCase();
    const matchesQuery = !query || haystack.includes(query);
    const matchesCategory = state.category === "toutes" || row.category === state.category;
    const matchesScored =
      !state.showOnlyScored || typeof row.globalScore === "number" || typeof row.fiabiliteScore === "number";
    const matchesAutonomy =
      state.autonomyThreshold <= 0 ||
      (typeof row.autonomieScore === "number" && row.autonomieScore >= state.autonomyThreshold);
    return matchesQuery && matchesCategory && matchesScored && matchesAutonomy;
  });

  rows.sort((a, b) => compareRows(a, b, state.sortKey, state.sortDir));
  return rows;
}

function compareRows(a, b, key, dir) {
  let va;
  let vb;
  switch (key) {
    case "brand":
      va = a.brand.toLowerCase();
      vb = b.brand.toLowerCase();
      break;
    case "model":
      va = a.model.toLowerCase();
      vb = b.model.toLowerCase();
      break;
    case "globalScore":
      va = typeof a.globalScore === "number" ? a.globalScore : -Infinity;
      vb = typeof b.globalScore === "number" ? b.globalScore : -Infinity;
      break;
    case "autonomieScore":
      va = typeof a.autonomieScore === "number" ? a.autonomieScore : -Infinity;
      vb = typeof b.autonomieScore === "number" ? b.autonomieScore : -Infinity;
      break;
    case "roadPerformanceScore":
      va = typeof a.roadPerformanceScore === "number" ? a.roadPerformanceScore : -Infinity;
      vb = typeof b.roadPerformanceScore === "number" ? b.roadPerformanceScore : -Infinity;
      break;
    case "offroadPerformanceScore":
      va = typeof a.offroadPerformanceScore === "number" ? a.offroadPerformanceScore : -Infinity;
      vb = typeof b.offroadPerformanceScore === "number" ? b.offroadPerformanceScore : -Infinity;
      break;
    case "utilityComfortScore":
      va = typeof a.utilityComfortScore === "number" ? a.utilityComfortScore : -Infinity;
      vb = typeof b.utilityComfortScore === "number" ? b.utilityComfortScore : -Infinity;
      break;
    case "privatePerformanceScore":
      va = typeof a.privatePerformanceScore === "number" ? a.privatePerformanceScore : -Infinity;
      vb = typeof b.privatePerformanceScore === "number" ? b.privatePerformanceScore : -Infinity;
      break;
    case "winterPerformanceScore":
      va = typeof a.winterPerformanceScore === "number" ? a.winterPerformanceScore : -Infinity;
      vb = typeof b.winterPerformanceScore === "number" ? b.winterPerformanceScore : -Infinity;
      break;
    case "fiabiliteScore":
      va = typeof a.fiabiliteScore === "number" ? a.fiabiliteScore : -Infinity;
      vb = typeof b.fiabiliteScore === "number" ? b.fiabiliteScore : -Infinity;
      break;
    default:
      va = typeof a.fiabiliteScore === "number" ? a.fiabiliteScore : -Infinity;
      vb = typeof b.fiabiliteScore === "number" ? b.fiabiliteScore : -Infinity;
  }

  let comparison;
  if (typeof va === "string" && typeof vb === "string") {
    comparison = va.localeCompare(vb, "fr", { sensitivity: "base" });
  } else {
    comparison = (va ?? 0) - (vb ?? 0);
  }

  return dir === "asc" ? comparison : -comparison;
}

function updateResultsCount(count) {
  const total = state.data.length;
  const suffix = count === total ? `${count} modèles` : `${count} / ${total} modèles`;
  resultsCount.textContent = count === 0 ? "Aucun résultat" : suffix;
}

function openScoreCellFromElement(cell) {
  if (!cell) return;
  const rowEl = cell.closest("tr");
  if (!rowEl) return;
  const rowId = rowEl.dataset.rowId;
  if (!rowId) return;
  const scoreKey = cell.dataset.scoreKey;
  if (!scoreKey) return;
  const rowData = state.data.find(item => item.id === rowId);
  if (!rowData) return;
  openScorePopover(rowData, scoreKey, cell);
}

function handleScoreCellClick(event) {
  const cell = event.target.closest(".score-cell");
  if (!cell || !tableBody.contains(cell)) return;
  openScoreCellFromElement(cell);
}

function handleScoreCellKeydown(event) {
  if (event.key !== "Enter" && event.key !== " ") return;
  const cell = event.target.closest(".score-cell");
  if (!cell) return;
  event.preventDefault();
  openScoreCellFromElement(cell);
}

function handleSortClick(event) {
  const key = event.currentTarget.dataset.key;
  if (!key) return;
  if (state.sortKey === key) {
    state.sortDir = state.sortDir === "asc" ? "desc" : "asc";
  } else {
    state.sortKey = key;
    state.sortDir = DESC_SORT_KEYS.has(key) ? "desc" : "asc";
  }
  updateSortButtons();
  render();
}

function formatMultiplier(value) {
  if (typeof value !== "number" || !Number.isFinite(value)) {
    return null;
  }
  const epsilon = 1e-6;
  if (Math.abs(value - Math.round(value)) < epsilon) {
    return String(Math.round(value));
  }
  if (Math.abs(value * 10 - Math.round(value * 10)) < epsilon) {
    return value.toFixed(1);
  }
  return value.toFixed(2);
}

function showWeightUpdateIndicator() {
  if (!weightUpdateIndicator) return;
  const message = "⚙ Pondérations ajustées selon vos usages";
  weightUpdateIndicator.textContent = message;
  weightUpdateIndicator.dataset.visible = "true";

  if (weightIndicatorTimeoutId) {
    window.clearTimeout(weightIndicatorTimeoutId);
  }

  weightIndicatorTimeoutId = window.setTimeout(() => {
    weightUpdateIndicator.dataset.visible = "false";
    weightIndicatorTimeoutId = null;
  }, 2000);
}

function haveMetricWeightsChanged(previousWeights, nextWeights) {
  if (!previousWeights || !nextWeights) {
    return false;
  }

  const keys = new Set([
    ...Object.keys(previousWeights),
    ...Object.keys(nextWeights),
  ]);

  for (const key of keys) {
    const previousValue = previousWeights[key];
    const nextValue = nextWeights[key];
    const previous =
      typeof previousValue === "number" && Number.isFinite(previousValue)
        ? previousValue
        : 1;
    const next =
      typeof nextValue === "number" && Number.isFinite(nextValue) ? nextValue : 1;

    if (Math.abs(previous - next) > MULTIPLIER_DISPLAY_THRESHOLD) {
      return true;
    }
  }

  return false;
}

function updateSortButtons() {
  const metricWeights = getActiveMetricWeightMap();
  const previousWeights = previousMetricWeights;
  let weightsChanged = false;

  if (sortButtons.length > 0) {
    const activeGlobalMetrics = new Set(getActiveGlobalMetricKeys());

    sortButtons.forEach(btn => {
      let baseLabel = btn.dataset.label;
      if (!baseLabel) {
        baseLabel = btn.textContent.trim();
        btn.dataset.label = baseLabel;
      }

      const key = btn.dataset.key;
      const isActive = key === state.sortKey;
      const isGlobalMetric = activeGlobalMetrics.has(key);
      const rawMultiplier = metricWeights?.[key];
      const multiplierIsNumber =
        typeof rawMultiplier === "number" && Number.isFinite(rawMultiplier);
      const shouldDisplayMultiplier =
        multiplierIsNumber &&
        Math.abs(rawMultiplier - 1) > MULTIPLIER_DISPLAY_THRESHOLD;
      const formattedMultiplier = shouldDisplayMultiplier
        ? formatMultiplier(rawMultiplier)
        : null;

      const previousValue = previousWeights?.[key];
      const hasPreviousValue =
        typeof previousValue === "number" && Number.isFinite(previousValue);
      const multiplierChanged =
        Boolean(previousWeights) &&
        multiplierIsNumber &&
        Math.abs((hasPreviousValue ? previousValue : 1) - rawMultiplier) >
          MULTIPLIER_DISPLAY_THRESHOLD;

      const ariaParts = [baseLabel];
      const fragments = [];

      const labelSpan = document.createElement("span");
      labelSpan.className = "sort-label";
      labelSpan.textContent = baseLabel;
      fragments.push(labelSpan);

      if (shouldDisplayMultiplier && formattedMultiplier) {
        const multiplierSpan = document.createElement("span");
        multiplierSpan.className = "sort-multiplier";
        multiplierSpan.textContent = `× ${formattedMultiplier}`;
        if (multiplierChanged) {
          multiplierSpan.classList.add("sort-multiplier--updated");
        }
        fragments.push(multiplierSpan);
        ariaParts.push(`pondération ${formattedMultiplier}`);
      }

      if (isActive) {
        const arrowSpan = document.createElement("span");
        arrowSpan.className = "sort-arrow";
        arrowSpan.textContent = state.sortDir === "asc" ? "▲" : "▼";
        fragments.push(arrowSpan);
        ariaParts.push(
          state.sortDir === "asc" ? "ordre croissant" : "ordre décroissant"
        );
      }

      btn.classList.toggle("active", isActive);
      btn.classList.toggle("global-metric", isGlobalMetric);
      btn.replaceChildren(...fragments);
      btn.setAttribute("aria-label", ariaParts.join(", "));

      if (multiplierChanged) {
        btn.classList.add("sort--weight-updated");
        btn.addEventListener(
          "animationend",
          () => {
            btn.classList.remove("sort--weight-updated");
          },
          { once: true }
        );
        weightsChanged = true;
      }
    });
  }

  if (!weightsChanged && haveMetricWeightsChanged(previousWeights, metricWeights)) {
    weightsChanged = true;
  }

  if (weightsChanged) {
    showWeightUpdateIndicator();
  }

  previousMetricWeights = metricWeights ? { ...metricWeights } : null;
}

function resetFilters() {
  invalidateMetricWeightCache();
  state.query = "";
  state.category = "toutes";
  state.showOnlyScored = false;
  state.autonomyThreshold = 0;
  state.sortKey = "globalScore";
  state.sortDir = "desc";
  state.selectedUseCases = new Set();
  state.selectedGlobalMetrics = [...DEFAULT_GLOBAL_METRIC_KEYS];
  state.data = cloneData(initialData);
  recomputeGlobalScores(state.selectedGlobalMetrics);
  updateUseCaseUI();
  updateUseCaseSummary(state.selectedGlobalMetrics);
  searchInput.value = "";
  categorySelect.value = "toutes";
  onlyScoredCheckbox.checked = false;
  updateAutonomyFilterUI();
  updateSortButtons();
  render();
}

searchInput.addEventListener("input", event => {
  state.query = event.target.value;
  render();
});

categorySelect.addEventListener("change", event => {
  state.category = event.target.value;
  render();
});

onlyScoredCheckbox.addEventListener("change", event => {
  state.showOnlyScored = event.target.checked;
  render();
});

if (autonomyThresholdInput && autonomyThresholdValue) {
  autonomyThresholdInput.addEventListener("input", event => {
    const value = Number(event.target.value);
    state.autonomyThreshold = Number.isFinite(value) ? value : 0;
    autonomyThresholdValue.textContent = state.autonomyThreshold.toFixed(1);
    updateAutonomySliderTrack();
    render();
  });
  updateAutonomyFilterUI();
}

sortButtons.forEach(btn => btn.addEventListener("click", handleSortClick));
resetButton.addEventListener("click", resetFilters);
tableBody.addEventListener("click", handleScoreCellClick);
tableBody.addEventListener("keydown", handleScoreCellKeydown);

if (scorePopover) {
  scorePopover.addEventListener("click", event => {
    if (event.target === scorePopover || event.target.dataset.popoverDismiss !== undefined) {
      closeScorePopover();
    }
  });
}

window.addEventListener("keydown", event => {
  if (event.key === "Escape") {
    if (scorePopover?.dataset.open === "true") {
      event.preventDefault();
      closeScorePopover();
      return;
    }
  }
});

renderUseCaseControls();
updateSortButtons();
init();

function updateAutonomySliderTrack() {
  if (!autonomyThresholdInput) return;
  const min = Number(autonomyThresholdInput.min ?? 0);
  const max = Number(autonomyThresholdInput.max ?? 100);
  const value = Number(autonomyThresholdInput.value ?? min);
  const range = max - min;
  const progress = range > 0 ? ((value - min) / range) * 100 : 0;
  const clamped = Math.min(100, Math.max(0, progress));
  autonomyThresholdInput.style.setProperty("--slider-fill", `${clamped}%`);
}

function updateAutonomyFilterUI() {
  if (!autonomyThresholdInput || !autonomyThresholdValue) return;
  autonomyThresholdInput.value = String(state.autonomyThreshold);
  autonomyThresholdValue.textContent = state.autonomyThreshold.toFixed(1);
  updateAutonomySliderTrack();
}
