type FilterDef = {
  key: string;
  label: string;
  weight: number;
  maxScore: number; // 10
  id: string; // clé dans component_quality.json
  category: string; // clé/chemin dans bikes.json
  explanation?: string;
};

type ComponentQualityTable = Record<string, number> & { _default?: number };
type ComponentQuality = Record<string, ComponentQualityTable>;

const pick = (obj: any, path: string) =>
  path.split('.').reduce((o, k) => (o == null ? undefined : o[k]), obj);

const norm = (v: unknown) => String(v ?? '').trim().toLowerCase();

export function resolveScore10(
  filter: FilterDef,
  bike: any,
  qualityTables: ComponentQuality
): number {
  if (!qualityTables || typeof qualityTables !== 'object') {
    return 0;
  }

  const rawValue = norm(pick(bike, filter.category) ?? bike?.[filter.category]);
  const table: ComponentQualityTable = qualityTables[filter.id] ?? { _default: 0 };
  const score = Object.prototype.hasOwnProperty.call(table, rawValue)
    ? table[rawValue]
    : typeof table._default === 'number'
      ? table._default
      : 0;

  // garde-fous
  if (typeof score !== 'number' || Number.isNaN(score)) return 0;
  return Math.max(0, Math.min(filter.maxScore, score));
}

// contribution pondérée au total
export function contributionPoints(
  filter: FilterDef,
  bike: any,
  quality: ComponentQuality
): number {
  const weight = Number.isFinite(filter.weight) ? filter.weight : 0;
  if (weight <= 0) return 0;

  const score10 = resolveScore10(filter, bike, quality);
  if (!Number.isFinite(score10) || score10 <= 0) {
    return 0;
  }

  const contribution = score10 * weight;
  if (!Number.isFinite(contribution)) {
    return 0;
  }

  const maxScore = Number.isFinite(filter.maxScore) ? filter.maxScore : 0;
  if (maxScore > 0) {
    const maxContribution = maxScore * weight;
    return Math.max(0, Math.min(maxContribution, contribution));
  }

  return Math.max(0, contribution);
}
